package com.almworks.jira.structure.api.util;

import com.almworks.jira.structure.api.forest.raw.Forest;

import java.util.HashMap;
import java.util.Map;

/**
 * A RowTree with an index from rowId to its {@link RowTree.Node}.
 */
public class IndexedRowTree extends RowTree {
  private final Map<Long, Node> myNodeMap;

  public IndexedRowTree(int size) {
    myNodeMap = new HashMap<>(size);
  }

  public IndexedRowTree() {
    this(16);
  }

  public static IndexedRowTree create(Forest forest) {
    IndexedRowTree r = new IndexedRowTree(forest.size());
    r.appendForest(forest);
    return r;
  }

  @Override
  protected Node createNode(long rowId, int flags) {
    Node node = super.createNode(rowId, flags);
    Node expunged = myNodeMap.put(rowId, node);
    assert expunged == null : "forest invariant broken: multiple nodes with id " + rowId;
    return node;
  }

  @Override
  protected void forgetNode(Node node) {
    super.forgetNode(node);
    myNodeMap.remove(node.getRowId());
    for (Node n = node.getFirstChild(); n != null; n = n.getNextSibling()) {
      forgetNode(n);
    }
  }

  /**
   * Quickly retrieves a node with the given row ID.
   *
   * @param rowId row ID
   * @return node or null if no such node
   */
  public Node getNodeWithRowId(long rowId) {
    return rowId == 0 ? getSuperRoot() : myNodeMap.get(rowId);
  }

  public boolean hasRow(long value) {
    return myNodeMap.containsKey(value);
  }
}
