package com.almworks.jira.structure.api.structure.history;

import com.almworks.integers.*;
import com.almworks.jira.structure.api.structure.history.HistoryEntry;
import com.almworks.jira.structure.api.item.ItemIdentity;
import com.almworks.jira.structure.api.util.ToString;
import com.google.common.collect.ImmutableSet;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.text.SimpleDateFormat;
import java.util.*;

public final class HistoryQuery {
  public static final List<HistoryEntry.Operation> ALL_OPERATIONS =
    Collections.unmodifiableList(Arrays.asList(HistoryEntry.Operation.values()));

  @Nullable
  public final Set<HistoryEntry.Operation> operations;

  @NotNull
  public final LongList isIssuesSorted;

  @NotNull
  public final LongList notIssuesSorted;

  @NotNull
  public final LongList isStructuresSorted;

  @NotNull
  public final LongList notStructuresSorted;

  @NotNull
  public final Set<String> isAuthors;

  @NotNull
  public final Set<String> notAuthors;

  @NotNull
  public final LongList isProjectsSorted;

  @NotNull
  public final LongList notProjectsSorted;

  @NotNull
  public final LongList isSynchronizersSorted;

  @NotNull
  public final LongList notSynchronizersSorted;

  @NotNull
  public final Set<ItemIdentity> isAncestors;

  @NotNull
  public final Set<ItemIdentity> notAncestors;

  @Nullable
  public final Long minDate;

  @Nullable
  public final Long maxDate;

  @Nullable
  public final Long startId;

  private volatile String myStringRepresentation;

  public HistoryQuery(
    Iterable<HistoryEntry.Operation> operations, LongList isIssuesSorted, LongList notIssuesSorted,
    LongList isStructuresSorted, LongList notStructuresSorted,
    Iterable<String> isAuthors, Iterable<String> notAuthors,
    LongList isProjectsSorted, LongList notProjectsSorted,
    LongList isSynchronizersSorted, LongList notSynchronizersSorted,
    Set<ItemIdentity> isAncestors, Set<ItemIdentity> notAncestors,
    Date minDate, Date maxDate, @Nullable Long startId)
  {
    Set<HistoryEntry.Operation> typeSet = set(operations);
    this.operations = typeSet.containsAll(ALL_OPERATIONS) ? null : typeSet;
    this.isIssuesSorted = nn(isIssuesSorted);
    this.notIssuesSorted = nn(notIssuesSorted);
    this.isStructuresSorted = nn(isStructuresSorted);
    this.notStructuresSorted = nn(notStructuresSorted);
    this.isAuthors = set(isAuthors);
    this.notAuthors = set(notAuthors);
    this.isProjectsSorted = nn(isProjectsSorted);
    this.notProjectsSorted = nn(notProjectsSorted);
    this.isSynchronizersSorted = nn(isSynchronizersSorted);
    this.notSynchronizersSorted = nn(notSynchronizersSorted);
    this.isAncestors = set(isAncestors);
    this.notAncestors = set(notAncestors);
    this.minDate = minDate != null ? minDate.getTime() : null;
    this.maxDate = maxDate != null ? maxDate.getTime() : null;
    this.startId = startId;
  }

  public HistoryQuery(long structureId, Long startId) {
    this(ALL_OPERATIONS, null, null, LongArray.create(structureId), null, null, null, null, null, null, null,
      null, null, null, null, startId);
  }

  private static LongList nn(LongList list) {
    return list == null ? LongList.EMPTY : list;
  }

  private static <T> Set<T> set(Iterable<T> it) {
    return it == null ? ImmutableSet.<T>of() : ImmutableSet.copyOf(it);
  }

  public String toString() {
    String s = myStringRepresentation;
    if (s != null) return s;
    StringBuilder sb = ToString.begin("query");
    append(sb, "types", operations);
    append(sb, "structures+", isStructuresSorted);
    append(sb, "structures-", notStructuresSorted);
    append(sb, "issues+", isIssuesSorted);
    append(sb, "issues-", notIssuesSorted);
    append(sb, "users+", isAuthors);
    append(sb, "users-", notAuthors);
    append(sb, "projects+", isProjectsSorted);
    append(sb, "projects-", notProjectsSorted);
    append(sb, "syncs+", isSynchronizersSorted);
    append(sb, "syncs-", notSynchronizersSorted);
    append(sb, "ancestors+", isAncestors);
    append(sb, "ancestors-", notAncestors);
    if (minDate != null || maxDate != null) {
      SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss", Locale.US);
      if (minDate != null) ToString.append(sb, "from", dateFormat.format(new Date(minDate)));
      if (maxDate != null) ToString.append(sb, "until", dateFormat.format(new Date(maxDate)));
    }
    if (startId != null) {
      sb.append("startId").append(ToString.EQ).append(startId);
    }
    s = ToString.end(sb);
    myStringRepresentation = s;
    return s;
  }

  private void append(StringBuilder sb, String name, Collection c) {
    if (c != null && !c.isEmpty()) {
      ToString.append(sb, name, c);
    }
  }

  private void append(StringBuilder sb, String name, LongList list) {
    if (list != null && !list.isEmpty()) {
      sb.append(name).append(ToString.EQ);
      String sep = "(";
      for (LongIterator ii : list) {
        sb.append(sep).append(ii.value());
        sep = ",";
      }
      sb.append(")").append(ToString.SEP);
    }
  }
}
