package com.almworks.jira.structure.api.row;

import com.almworks.integers.*;
import com.almworks.jira.structure.api.item.CoreItemTypes;
import com.almworks.jira.structure.api.util.LongFilteringIterator;
import org.jetbrains.annotations.Nullable;

public class StructureRows {
  /**
   * Temporary rows have negative values and start from -100 going down to Long.MIN_VALUE.
   */
  public static final long STARTING_TEMPORARY_ROW_ID = -100;

  private StructureRows() {}

  public static void validateForestRowId(long rowId) {
    String error = getRowIdError(rowId);
    if (error != null) {
      throw new IllegalArgumentException(error);
    }
  }

  public static void validateForestRowIds(@Nullable LongList rows) {
    if (rows != null) {
      for (LongListIterator ii = rows.iterator(); ii.hasNext(); ) {
        String error = getRowIdError(ii.nextValue());
        if (error != null) {
          throw new IllegalArgumentException("[" + ii.index() + "] " + error);
        }
      }
    }
  }

  public static LongIterator withoutSuperRoot(LongIterator rowIterator) {
    return new LongFilteringIterator(rowIterator, row -> row != SuperRootRow.SUPER_ROOT_ROW_ID);
  }

  public static LongIterable withoutSuperRoot(LongIterable rowIterable) {
    return () -> withoutSuperRoot(rowIterable.iterator());
  }

  public static boolean isSuperRoot(long rowId) {
    return rowId == SuperRootRow.SUPER_ROOT_ROW_ID;
  }

  public static boolean isSuperRoot(StructureRow row) {
    return row != null && row.getRowId() == SuperRootRow.SUPER_ROOT_ROW_ID;
  }

  private static String getRowIdError(long rowId) {
    if (rowId == 0) {
      return "row ID must be non-zero";
    }
    if (rowId == SuperRootRow.SUPER_ROOT_ROW_ID) {
      return "row ID " + SuperRootRow.SUPER_ROOT_ROW_ID + " is reserved for super-root and cannot be used";
    }
    return null;
  }

  public static boolean isGenerator(StructureRow row) {
    return isItemType(row, CoreItemTypes.GENERATOR);
  }

  public static boolean isEffector(StructureRow row) {
    return isItemType(row, CoreItemTypes.EFFECTOR);
  }

  public static boolean isAutomation(StructureRow row) {
    return isGenerator(row) || isEffector(row);
  }

  public static boolean isLoopMarker(StructureRow row) {
    return isItemType(row, CoreItemTypes.LOOP_MARKER);
  }

  public static boolean isItemType(StructureRow row, String itemType) {
    return itemType != null && row != null && itemType.equals(row.getItemId().getItemType());
  }

  public static boolean isTemporary(long rowId) {
    return rowId <= STARTING_TEMPORARY_ROW_ID;
  }

  public static long temporaryRow(int sequence) {
    if (sequence < 0) throw new IllegalArgumentException("sequence " + sequence);
    return STARTING_TEMPORARY_ROW_ID - sequence;
  }

  public static boolean isItemVisibleForSure(StructureRow row) {
    return TransientRow.getDelegate(row) instanceof SimpleRow.Unchecked;
  }

  public static StructureRow makeShallow(StructureRow row) {
    return TransientRow.replaceDelegate(row, ShallowRow::copy);
  }

  public static StructureRow makeUnchecked(StructureRow row) {
    return TransientRow.replaceDelegate(row, SimpleRow::tryMakeUnchecked);
  }
}
