package com.almworks.jira.structure.api.item;

import com.atlassian.annotations.Internal;
import com.atlassian.annotations.PublicApi;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.function.BiConsumer;

@PublicApi
public interface ItemResolver {
  @Nullable
  <T> T resolveItem(@Nullable ItemIdentity itemId, @NotNull Class<T> itemClass);

  @Internal
  @Nullable
  Object resolveUnchecked(@Nullable ItemIdentity itemId);

  /**
   * Resolves all the items from the given id collection and passes them to consumer.
   * Item retrieving is done with a permission check. Bulk resolve is used when available.
   * If an item is unresolved for any reason, including that the item is unavailable for the current user, null is passed to the consumer.
   * Thus, all the items are passed to the consumer together with the corresponding item id.
   *
   * @param itemIds   id collection of items to retrieve
   * @param itemClass expected item class
   * @param consumer  handler for (item id, item) pairs
   * @param <T>       type of expected item class
   */
  <T> void resolveItems(@NotNull Collection<ItemIdentity> itemIds, @NotNull Class<T> itemClass, @NotNull BiConsumer<ItemIdentity, T> consumer);


  /**
   * Resolves all the items from the given id collection and passes them to consumer.
   * No permission check is performed here. Bulk resolve is used when available.
   * If an item is unresolved for any reason, null is passed to the consumer.
   * Thus, all the items are passed to the consumer together with the corresponding item id.
   *
   * @param itemIds  id collection of items to retrieve
   * @param consumer handler for (item id, item) pairs
   */
  void resolveItemsUnchecked(@NotNull Collection<ItemIdentity> itemIds, @NotNull BiConsumer<ItemIdentity, Object> consumer);
}
