package com.almworks.jira.structure.api.cache;

import com.atlassian.annotations.Internal;
import com.atlassian.jira.user.ApplicationUser;
import org.jetbrains.annotations.NotNull;

@Internal
public interface StructureCacheHelper {
  /**
   * <p>Clears all caches on the current JIRA instance that are maintained by Structure and Structure extensions.</p>
   *
   * <p>Note that in JIRA Data Center environment, this does not clear caches on other nodes.</p>
   *
   * <p>After this method has finished execution, the caches are cleared.</p>
   *
   * @see com.almworks.jira.structure.api.lifecycle.CachingComponent
   * @see #clearAllStructureCachesOnAllNodes()
   */
  void clearAllStructureCaches();

  /**
   * <p>Clears all caches (maintained by Structure and Structure extensions) on the current JIRA instance and causes
   * eventual caches clearing on other nodes (when running on JIRA Data Center).</p>
   *
   * <p>When running on non-JDC instance, works the same as {@link #clearAllStructureCaches()}.</p>
   *
   * <p>After this method has finished execution, the caches are cleared on the local node, but may not be cleared
   * yet on other nodes.</p>
   *
   * @see com.almworks.jira.structure.api.lifecycle.CachingComponent
   * @see #clearAllStructureCaches()
   */
  void clearAllStructureCachesOnAllNodes();

  /**
   * <p>Clears caches on the all nodes for the given non-anonymous user (maintained by Structure and Structure extensions) on the current
   * JIRA instance and causes eventual caches clearing on other nodes (when running on JIRA Data Center).</p>
   *
   * <p>When running on non-JDC instance, works the same as {@link #clearUserStructureCaches(ApplicationUser)}.</p>
   *
   * <p>After this method has finished execution, the user-related caches are cleared on the local node, but may not be cleared yet on
   * other nodes.</p>
   *
   * @see com.almworks.jira.structure.api.lifecycle.CachingComponent
   * @see #clearUserStructureCaches(ApplicationUser)
   */
  default void clearUserStructureCachesOnAllNodes(@NotNull ApplicationUser user) {
    clearUserStructureCachesOnAllNodes(user.getKey());
  }

  /**
   * <p>Clears caches on the all nodes for the given user key (maintained by Structure and Structure extensions) on the current
   * JIRA instance and causes eventual caches clearing on other nodes (when running on JIRA Data Center).</p>
   *
   * <p>After this method has finished execution, the user-related caches are cleared on the local node, but may not be cleared yet on
   * other nodes.</p>
   *
   * <p>Implementation should guarantee that cache will be cleared even user associated with the key doesn't exist, that might happen
   * in case of changing user key during anonymization for the user that was created prior Jira 8.3</p>
   */
  void clearUserStructureCachesOnAllNodes(@NotNull String userKey);

  /**
   * Clears caches on the current node for the current user.
   */
  void clearCurrentUserStructureCaches();

  /**
   * Clears caches on the current node for the given non-anonymous user.
   */
  void clearUserStructureCaches(@NotNull ApplicationUser user);
}
