package com.almworks.jira.structure.api.attribute.loader.builder;

import com.almworks.jira.structure.api.attribute.AttributeSpec;
import com.almworks.jira.structure.api.attribute.loader.*;
import com.almworks.jira.structure.api.attribute.loader.basic.SimpleDerivedAttributeLoader;
import com.almworks.jira.structure.api.item.ItemIdentity;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Set;
import java.util.function.BiFunction;
import java.util.function.Function;

public class UniDerivedAttributeLoaderBuilder<T, D> extends AttributeLoaderBuilder<T, UniDerivedAttributeLoaderBuilder<T, D>> {
  private AttributeSpec<D> myDependencySpec;
  private BiFunction<? super D, DerivedAttributeContext, ? extends T> myValueFunction;
  private Function<D, ItemIdentity> myTrailFunction;
  private boolean myYieldOnNull = true;

  public SimpleDerivedAttributeLoader<T, D> build() {
    return new BuiltDerivedLoader<>(
      notNull(myAttributeSpec, "attributeSpec"),
      notNull(myDependencySpec, "dependencySpec"),
      myYieldOnNull,
      notNull(myValueFunction, "derivationFunction"),
      myTrailFunction,
      nullableCollectionOfNonNulls(buildContextDependencies(), "contextDependencies"),
      myCachingStrategy,
      myGlobalTrail);
  }
  
  public UniDerivedAttributeLoaderBuilder<T, D> dependency(AttributeSpec<D> dependencySpec) {
    myDependencySpec = dependencySpec;
    return this;
  }

  public UniDerivedAttributeLoaderBuilder<T, D> valueFunction(BiFunction<? super D, DerivedAttributeContext, ? extends T> valueFunction) {
    myValueFunction = valueFunction;
    return this;
  }

  public UniDerivedAttributeLoaderBuilder<T, D> valueFunction(Function<? super D, ? extends T> valueFunction) {
    return valueFunction((d, ctx) -> valueFunction.apply(d));
  }

  public UniDerivedAttributeLoaderBuilder<T, D> trail(Function<D, ItemIdentity> trailFunction) {
    myTrailFunction = trailFunction;
    if (trailFunction != null) contextDependency(AttributeContextDependency.TRAIL);
    return this;
  }

  public UniDerivedAttributeLoaderBuilder<T, D> yieldOnNull(boolean yieldOnNull) {
    myYieldOnNull = yieldOnNull;
    return this;
  }


  public static class BuiltDerivedLoader<T, D> extends SimpleDerivedAttributeLoader<T, D> {
    private final BiFunction<? super D, DerivedAttributeContext, ? extends T> myDerivationFunction;
    private final Function<D, ItemIdentity> myTrailFunction;
    private final Set<AttributeContextDependency> myContextDependencies;
    private final AttributeCachingStrategy myCachingStrategy;
    private final TrailItemSet myGlobalTrail;

    public BuiltDerivedLoader(AttributeSpec<T> spec, AttributeSpec<? extends D> dependency, boolean yield,
      BiFunction<? super D, DerivedAttributeContext, ? extends T> derivationFunction, Function<D, ItemIdentity> trailFunction,
      Set<AttributeContextDependency> contextDependencies,
      AttributeCachingStrategy cachingStrategy, TrailItemSet globalTrail)
    {
      super(spec, dependency, yield);
      myDerivationFunction = derivationFunction;
      myTrailFunction = trailFunction;
      myContextDependencies = contextDependencies;
      myCachingStrategy = cachingStrategy;
      myGlobalTrail = globalTrail;
    }

    @Nullable
    @Override
    protected T getValue(@NotNull D value, DerivedAttributeContext context) {
      if (myTrailFunction != null) {
        ItemIdentity trail = myTrailFunction.apply(value);
        if (trail != null) {
          context.addTrail(trail);
        }
      }
      return myDerivationFunction.apply(value, context);
    }

    @Nullable
    @Override
    public Set<AttributeContextDependency> getContextDependencies() {
      return myContextDependencies == null ? super.getContextDependencies() : myContextDependencies;
    }

    @Nullable
    @Override
    public AttributeCachingStrategy getCachingStrategy() {
      return myCachingStrategy == null ? super.getCachingStrategy() : myCachingStrategy;
    }

    @Nullable
    @Override
    public TrailItemSet getGlobalTrail() {
      return myGlobalTrail == null ? super.getGlobalTrail() : myGlobalTrail;
    }
  }
}
