package com.almworks.jira.structure.api.attribute.loader;

import com.almworks.jira.structure.api.row.StructureRow;
import com.almworks.jira.structure.api.row.SuperRootRow;
import com.atlassian.annotations.PublicApi;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.List;

/**
 * <p>The context interface for propagate attribute loaders.</p>
 *
 * <p>This interface is passed to the function that loads the child row's value. When
 * that function is created with {@link PropagateAttributeLoader#loadChildren} function,
 * a different context is used - {@link PropagateAttributeContext.Parent}.</p>
 *
 * @see PropagateAttributeLoader
 * @see PropagateAttributeContext.Parent
 */
@PublicApi
public interface PropagateAttributeContext extends RowAttributeContext {
  /**
   * Returns the index of the currently calculated row in the list of the parent's children.
   *
   * @return the index of the current row among siblings
   */
  int getIndex();

  /**
   * <p>The context interface for preparing children loading function by a propagate.</p>
   *
   * <p>Note that this interface does not extend {@link AttributeLoaderContext}, so no
   * dependency values are available at this point. It can be used by an implementation of
   * a {@link PropagateAttributeLoader} to get a list of children and otherwise prepare
   * the actual loading function, which will be passed {@link PropagateAttributeContext}.</p>
   *
   * @see PropagateAttributeContext
   * @see PropagateAttributeLoader
   */
  @PublicApi
  interface Parent extends AttributeContext {
    /**
     * Returns the list of all children of the parent row. The values for Some of these children are
     * about to be calculated.
     *
     * @return children rows under the parent with the indicated value
     */
    @NotNull
    List<StructureRow> getChildren();

    /**
     * <p>Returns the current parent row, which children are about to be calculated.</p>
     *
     * <p>Null means the children is only the super-root. If this method returns a super-root,
     * then the children will be roots. Note that this logic does not depend on the value of
     * {@link PropagateAttributeLoader#isLoadingSuperRoot()}.</p>
     *
     * @see PropagateAttributeLoader
     */
    @Nullable
    StructureRow getParent();

    /**
     * Returns true if the current parent is nothing and the child to be loaded is the super-root.
     *
     * @return true if the super-root is about to be loaded
     */
    default boolean isLoadingSuperRoot() {
      return getParent() == null;
    }

    /**
     * Returns true if the current parent is the super-root and the children are forest roots.
     *
     * @return true if forest roots are about to be loaded
     */
    default boolean isLoadingRoots() {
      StructureRow parent = getParent();
      return parent != null && parent.getRowId() == SuperRootRow.SUPER_ROOT_ROW_ID;
    }
  }
}
