package com.almworks.jira.structure.api.forest;

import com.almworks.jira.structure.api.auth.StructureAuth;
import com.almworks.jira.structure.api.error.StructureException;
import com.almworks.jira.structure.api.rest.*;
import com.almworks.jira.structure.api.structure.history.HistoryService;
import com.almworks.jira.structure.api.util.*;
import com.atlassian.annotations.Internal;
import com.atlassian.annotations.PublicApi;
import com.atlassian.jira.component.ComponentAccessor;
import com.atlassian.jira.user.ApplicationUser;
import com.atlassian.jira.web.HttpServletVariables;
import org.apache.commons.lang.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import javax.annotation.concurrent.Immutable;
import javax.servlet.http.HttpServletRequest;
import java.util.*;

/**
 * <p>{@code ForestSpec} is used to identify structured content. All content that can be displayed by the
 * Structure Widget in the Grid panels &mdash; structures, queries and others &mdash; is identified by {@code ForestSpec}.</p>
 *
 * <p>An instance of {@code ForestSpec} can be created through one of the factory methods, or deserialized from JSON.
 * It then can be fed to {@link ForestService} to obtain {@link ForestSource}. The {@code ForestSource} can be
 * asked for an update of the forest content, based on the last version that the caller has observed and cached.</p>
 *
 * <p>Unlike {@code ForestSource}, instances of {@code ForestSpec} can be stored for a long time.</p>
 *
 * <p>{@code ForestSpec} parameters belong to two groups: base content specification and adjustment specification.
 * A {@code ForestSource} will first produce the base content and then will make additional adjustments to it,
 * according to the spec.</p>
 *
 * <h3>Base content specification</h3>
 *
 * <p>Base content can be of the following types, each having its own factory method:</p>
 * <ul>
 *   <li>A specific structure &mdash; see {@link #structure(long)}</li>
 *   <li>A query &mdash; see {@link #sQuery(String, String)}</li>
 *   <li>A specific version of a structure &mdash; see {@link #version(long, int)}</li>
 *   <li>A user's clipboard &mdash; see {@link #clipboard(String)}</li>
 * </ul>
 *
 * <h3>Adjustments</h3>
 *
 * <p>The following parameters adjust the produced content:</p>
 * <ul>
 *   <li>Transformations apply generators to the base content &mdash; see {@link #transform(String, Map)}. This is what
 *   Structure uses in the transformations and quick transformations panels.</li>
 *   <li>"Secured" parameter specifies that the content must be filtered for access and all sub-trees with the
 *   items that the user cannot see must be removed. <strong>Very important</strong> &mdash; {@link #secure(String)}</li>
 *   <li>Setting {@code title} parameter to true would make the content placed under the top-level "title" item
 *   &mdash; {@link Builder#setTitle(boolean)}</li>
 *   <li>Setting {@code skeleton} parameter to true would prevent Structure engine from generating the dynamic
 *   content. The result would be a forest with only static content, including generators, but without generators'
 *   action &mdash; {@link #skeleton(long)}</li>
 * </ul>
 *
 * <p>Additionally, {@code userKey} parameter specifies the account, which is used to:</p>
 * <ul>
 *   <li>secure the forest, if {@code secured} parameter is set, and</li>
 *   <li>run transformations under, if {@code transformations} are specified.</li>
 * </ul>
 *
 * <p>Note that there are no permission checks when {@code ForestSpec} is created. Permissions are checked only
 * when {@link ForestService#getForestSource(ForestSpec)} is called.</p>
 *
 * @see ForestService
 * @see Builder
 */
@PublicApi
@Immutable
public final class ForestSpec {
  @Nullable
  private final Long myStructureId;

  @Nullable
  private final Integer myVersion;

  @Nullable
  private final SQuery mySQuery;

  @Nullable
  private final String myClipboardSessionId;

  @Nullable
  private final String myUserKey;

  @Nullable
  private final List<Transformation> myTransformations;

  private final boolean mySecured;
  
  private final boolean mySkeleton;

  private final boolean myTitle;

  private int myHashCode;

  private ForestSpec(@Nullable Long structureId, @Nullable Integer version, @Nullable SQuery sQuery,
    @Nullable String clipboardSessionId, @Nullable List<Transformation> transformations, @Nullable String userKey,
    boolean secured, boolean skeleton, boolean title)
  {
    assert StructureUtil.onlyOneIsTrue(structureId != null, sQuery != null, clipboardSessionId != null);
    assert structureId != null || version == null;
    assert !(skeleton && (version != null || sQuery != null || clipboardSessionId != null || transformations != null));
    myStructureId = structureId;
    myVersion = version;
    mySQuery = sQuery;
    myClipboardSessionId = clipboardSessionId;
    if (transformations != null && !transformations.isEmpty()) {
      myTransformations = Collections.unmodifiableList(transformations);
    } else {
      myTransformations = null;
    }
    if (myStructureId == null || myTransformations != null || secured) {
      myUserKey = userKey;
    } else {
      myUserKey = null;
    }
    mySecured = secured;
    mySkeleton = skeleton;
    myTitle = title;
  }

  /**
   * Creates a new builder for {@code ForestSpec}.
   */
  @NotNull
  public static Builder builder() {
    return new Builder();
  }

  /**
   * Creates a new build that can be used to adjust a given {@code ForestSpec}.
   */
  @NotNull
  public static Builder builder(@Nullable ForestSpec spec) {
    return spec == null ? new Builder() : new Builder(spec);
  }

  /**
   * <p>Constructs {@code ForestSpec} that represents a structure. The resulting content would display the specified
   * structure with all dynamic content properly calculated.</p>
   *
   * <p>The resulting forest spec is secured for the current user. The forests produced by the corresponding
   * forest source will exclude sub-trees that do not contain items visible to the user.</p>
   *
   * @param structureId structure ID
   * @return forest spec
   * @see #structure(long, ApplicationUser)
   * @see #unsecuredStructure(long)
   * @see ForestService#getForestSource(ForestSpec)
   * @see StructureAuth
   */
  @NotNull
  public static ForestSpec structure(long structureId) {
    return structure(structureId, StructureAuth.getUser());
  }

  /**
   * <p>Constructs {@code ForestSpec} that represents a structure. The resulting content would display the specified
   * structure with all dynamic content properly calculated.</p>
   *
   * <p>The resulting forest spec is secured for the specified user. The forests produced by the corresponding
   * forest source will exclude sub-trees that do not contain items visible to the user.</p>
   *
   * @param structureId structure ID
   * @return forest spec
   * @see #structure(long)
   * @see #unsecuredStructure(long)
   * @see ForestService#getForestSource(ForestSpec)
   */
  @NotNull
  public static ForestSpec structure(long structureId, @Nullable ApplicationUser user) {
    return builder()
      .setStructureId(structureId)
      .setUserKey(JiraUsers.getKeyFor(user))
      .setSecured(true)
      .build();
  }

  /**
   * <p>Constructs {@code ForestSpec} that represents a structure. The resulting content would display the specified
   * structure with all dynamic content properly calculated.</p>
   *
   * <p><strong>The resulting forest spec is unsecured.</strong> Use {@link #secure(String)} or {@link #structure(long, ApplicationUser)}
   * to properly filter the content for a given user.</p>
   *
   * @param structureId structure ID
   * @return forest spec
   * @see #structure(long)
   * @see #structure(long, ApplicationUser)
   */
  @NotNull
  public static ForestSpec unsecuredStructure(long structureId) {
    return builder().setStructureId(structureId).build();
  }

  /**
   * <p>Constructs {@code ForestSpec} that represents a structure with no dynamic content. The resulting content would
   * show only static rows of the structure, including generators, but generators will not have acted.</p>
   *
   * <p><strong>The resulting forest spec is unsecured.</strong> Use {@link #secure(String)} to properly filter the
   * content for a given user.</p>
   *
   * @param structureId structure ID
   * @return forest spec
   */
  @NotNull
  public static ForestSpec skeleton(long structureId) {
    return builder().setStructureId(structureId).setSkeleton(true).build();
  }

  /**
   * <p>Constructs {@code ForestSpec} that represents a historical version of a structure. The resulting content
   * would only display the manually added rows &mdash; automation does not work on historical versions.</p>
   *
   * <p><strong>The resulting forest spec is unsecured.</strong> Use {@link #secure(String)} to properly filter the
   * content for a given user.</p>
   *
   * @param structureId structure ID
   * @param version version number
   * @return forest spec
   * @see HistoryService
   */
  @NotNull
  public static ForestSpec version(long structureId, int version) {
    return builder().setStructureId(structureId).setVersion(version).build();
  }

  /**
   * <p>Constructs {@code ForestSpec} that represents a structured query. A structured query is a textual, human-readable
   * definition of the content.</p>
   *
   * <p>There may be different types of SQuery. So far, the following are supported:</p>
   * <ul>
   *   <li><strong>{@code jql}</strong> query produces a flat structure with results of JQL query (up to 1000 issues).</li>
   *   <li><strong>{@code text}</strong> query produces a flat structure with results of text-based JQL query (up to 1000 issues).</li>
   *   <li><strong>{@code cql}</strong> query works only if Structure.Pages is installed and configured, it produces
   *   a flat structure with results of CQL query in all connected Confluence instances (up to 1000 pages).</li>
   * </ul>
   *
   * <p>The resulting forest spec is secured for the current user (see also {@link StructureAuth}). The result will be filtered
   * according to the user's access to the items. Use {@link #getUnsecured()} to get an unsecured version of the
   * result, or {@link #secure(String)} to secure it for another user.</p>
   *
   * @param type type of query
   * @param query query text
   * @return forest spec
   */
  @NotNull
  public static ForestSpec sQuery(@NotNull String type, @NotNull String query) {
    return builder()
      .setSQuery(type, query)
      .setSecured(true)
      .setCurrentUserKey()
      .build();
  }

  /**
   * <p>Constructs {@code ForestSpec} that represents a user's clipboard. The clipboard's lifecycle is tied to
   * the user's session, so we're using session ID to identify the user.</p>
   *
   * @param sessionId session ID from servlet subsystem
   * @return forest spec
   * @see HttpServletRequest#getSession()
   */
  @NotNull
  public static ForestSpec clipboard(@NotNull String sessionId) {
    return builder().setClipboardSessionId(sessionId).build();
  }

  /**
   * <p>Creates a new {@code ForestSpec} that is secured for the specified user. The rows will be checked and
   * all sub-trees that contain only the items that the user does not have read access to will be removed.</p>
   *
   * @param userKey user key
   * @return adjusted forest spec
   */
  @NotNull
  public ForestSpec secure(@Nullable String userKey) {
    return builder(this).setUserKey(userKey).setSecured(true).build();
  }

  /**
   * <p>Creates a transformed {@code ForestSpec} that adds a single transformation to the result of the original
   * forest spec. The transformation is specified by the module key of the generator and the parameters
   * map.</p>
   *
   * <p>The new {@code ForestSpec} will have the current user as its {@code userKey} parameter. Note that this
   * is the user account under which the generators will execute, when performing <strong>all</strong> transformations,
   * not only the most recent one.</p>
   *
   * <p>The resulting forest spec's "secured" flag will be the same as {@link #isSecured()} for this forest spec.</p>
   *
   * @param module complete plugin module key of the generator that should perform the transformation
   * @param params parameters for the generator
   * @return adjusted forest spec
   * @see com.almworks.jira.structure.api.generator.CoreStructureGenerators
   * @see com.almworks.jira.structure.api.generator.CoreGeneratorParameters
   */
  @NotNull
  public ForestSpec transform(@NotNull String module, @Nullable Map<String, Object> params) {
    assert !mySkeleton;
    return builder(this).setCurrentUserKey().addTransformation(module, params).setSkeleton(false).build();
  }

  /**
   * Returns true if this forest spec contains transformation.
   */
  public boolean isTransformed() {
    return myTransformations != null && !myTransformations.isEmpty();
  }

  /**
   * For a forest spec with transformations, returns the forest spec without the last used transformation.
   * If the forest spec does not contain transformations, returns null.
   *
   * @see #getLastTransformation()
   */
  @Nullable
  public ForestSpec getLastTransformedSpec() {
    if (!isTransformed()) {
      return null;
    }
    assert myTransformations != null && !myTransformations.isEmpty();
    return builder(this).setTransformations(myTransformations.subList(0, myTransformations.size() - 1)).setSkeleton(false).build();
  }

  /**
   * Returns the last transformation used by this forest spec. Returns null if there are no transformations.
   *
   * @see #getLastTransformedSpec()
   */
  @Nullable
  public Transformation getLastTransformation() {
    if (!isTransformed()) {
      return null;
    }
    assert myTransformations != null && !myTransformations.isEmpty();
    return myTransformations.get(myTransformations.size() - 1);
  }

  /**
   * Returns the same forest spec but without transformations.
   */
  @NotNull
  public ForestSpec getUntransformedSpec() {
    return isTransformed() ? builder(this).setTransformations(null).build() : this;
  }

  /**
   * Returns structure ID if the base content is a structure or a structure version, null otherwise.
   */
  @Nullable
  public Long getStructureId() {
    return myStructureId;
  }

  /**
   * Returns version if the base content is a structure version, null otherwise.
   */
  @Nullable
  public Integer getVersion() {
    return myVersion;
  }

  /**
   * Returns {@link SQuery} if the base content is a SQuery, null otherwise.
   */
  @Nullable
  public SQuery getSQuery() {
    return mySQuery;
  }

  /**
   * Returns session ID used to identify the clipboard if the base content is a clipboard, null otherwise.
   */
  @Nullable
  public String getClipboardSessionId() {
    return myClipboardSessionId;
  }

  /**
   * Returns user key that is used to run transformations and to secure the forest, if corresponding
   * options are present.
   */
  @Nullable
  public String getUserKey() {
    return myUserKey;
  }

  /**
   * Returns the list of transformations applied to the base forest.
   */
  @NotNull
  public List<Transformation> getTransformations() {
    return myTransformations == null ? Collections.<Transformation>emptyList() : myTransformations;
  }

  /**
   * Returns true if the result will be secured, that is, the items in the forest will be checked for being accessible
   * to the user and fully inaccessible sub-trees will be removed.
   */
  public boolean isSecured() {
    return mySecured;
  }

  /**
   * Returns true if forest spec produces a base forest without running generators.
   */
  public boolean isSkeleton() {
    return mySkeleton;
  }

  /**
   * Returns true if forest spec has {@code title} option on.
   */
  public boolean hasTitle() {
    return myTitle;
  }

  /**
   * Returns the same forest spec, but with {@code secured} flag turned off.
   */
  @NotNull
  public ForestSpec getUnsecured() {
    return isSecured() ? builder(this).setSecured(false).build() : this;
  }

  /**
   * Returns the same forest spec but without {@code title} option.
   */
  @NotNull
  public ForestSpec withoutTitle() {
    if (!myTitle) return this;
    return builder(this).setTitle(false).build();
  }

  /**
   * Returns the same forest spec but with {@code title} option.
   */
  @NotNull
  public ForestSpec withTitle() {
    if (myTitle) return this;
    return builder(this).setTitle(true).build();
  }

  /**
   * Applies {@link Visitor} to this spec.
   */
  public <T> T accept(Visitor<T> visitor) throws StructureException {
    if (isSecured()) {
      return visitor.visitSecured(this);
    }
    if (isTransformed()) {
      return visitor.visitTransformation(this);
    }
    if (myStructureId != null) {
      return visitor.visitStructure(this);
    }
    if (mySQuery != null) {
      return visitor.visitSQuery(this);
    }
    if (myClipboardSessionId != null) {
      return visitor.visitClipboard(this);
    }
    throw new UnsupportedOperationException();
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;
    ForestSpec that = (ForestSpec) o;
    return mySecured == that.mySecured &&
      mySkeleton == that.mySkeleton &&
      myTitle == that.myTitle &&
      Objects.equals(myStructureId, that.myStructureId) &&
      Objects.equals(myVersion, that.myVersion) &&
      Objects.equals(mySQuery, that.mySQuery) &&
      Objects.equals(myClipboardSessionId, that.myClipboardSessionId) &&
      Objects.equals(myUserKey, that.myUserKey) &&
      Objects.equals(myTransformations, that.myTransformations);
  }

  @Override
  public int hashCode() {
    int hashCode = myHashCode;
    if (hashCode == 0) {
      hashCode = Objects.hash(myStructureId, myVersion, mySQuery, myClipboardSessionId,
        myUserKey, myTransformations, mySecured, mySkeleton, myTitle);
      if (hashCode == 0) {
        hashCode++;
      }
      myHashCode = hashCode;
    }
    return hashCode;
  }

  @Override
  public String toString() {
    StringBuilder r = new StringBuilder().append(StructureUtil.toJson(toRest()));
    if (mySkeleton) {
      r.append("(skeleton)");
    }
    if (myUserKey != null) {
      r.append("@").append(myUserKey);
    }
    if (mySecured) {
      r.append("(secure)");
    }
    return r.toString();
  }

  /**
   * This method can be used to restore a forest spec from a REST transfer object.
   *
   * @param rfs REST forest spec
   * @return ForestSpec
   * @throws IllegalArgumentException if the transfer object is invalid
   */
  @NotNull
  public static ForestSpec fromRest(@Nullable RestForestSpec rfs) throws IllegalArgumentException {
    if (rfs == null) {
      throw new IllegalArgumentException("Forest spec is required");
    }
    Long structureId = rfs.structureId;
    Integer version = rfs.version;
    SQuery sQuery = SQuery.fromRest(rfs);
    String sessionId = null;
    boolean secured = true;
    boolean clipboard = "clipboard".equals(rfs.type);
    boolean title = rfs.title != null && rfs.title;

    if (!StructureUtil.onlyOneIsTrue(structureId != null, sQuery != null, clipboard)) {
      throw new IllegalArgumentException("Exactly one of structureId, sQuery, or clipboard must be present");
    }
    if (structureId == null && version != null) {
      throw new IllegalArgumentException("Version can be specified only with structureId");
    }
    if (title && structureId == null) {
      throw new IllegalArgumentException("Title item is supported only for structureId");
    }

    if (clipboard) {
      final HttpServletVariables variables = ComponentAccessor.getComponent(HttpServletVariables.class); // TCCL ok
      sessionId = JiraComponents.withThreadContextClassLoaderOf(variables,
        (SimpleCallable<String>) () -> {
          HttpServletRequest request = variables != null ? variables.getHttpRequest() : null;
          if (request != null) {
            return request.getSession().getId();
          } else {
            throw new IllegalStateException("Cannot obtain http session to create clipboard spec");
          }
        });
      secured = false;
    }

    List<Transformation> transformations = null;
    if (rfs.transforms != null && !rfs.transforms.isEmpty()) {
      transformations = new ArrayList<>();
      for (RestTransformSpec rts : rfs.transforms) {
        transformations.add(new Transformation(rts.module, rts.params));
      }
    }

    return new ForestSpec(structureId, version, sQuery, sessionId, transformations, StructureAuth.getUserKey(), secured,
      false, title);
  }

  /**
   * Produces REST forest specification based on this instance's parameters.
   */
  @NotNull
  public RestForestSpec toRest() {
    RestForestSpec rfs = new RestForestSpec();
    rfs.structureId = myStructureId;
    rfs.version = myVersion;
    if (mySQuery != null) {
      rfs.sQuery = new RestSQuery();
      rfs.sQuery.query = mySQuery.getQuery();
      rfs.sQuery.type = mySQuery.getType();
    }
    if (myClipboardSessionId != null) {
      rfs.type = "clipboard";
    }
    if (myTransformations != null) {
      rfs.transforms = new ArrayList<>(myTransformations.size());
      for (Transformation t : myTransformations) {
        RestTransformSpec rts = new RestTransformSpec();
        rts.module = t.getModule();
        rts.params = t.getParams();
        rfs.transforms.add(rts);
      }
    }
    if (myTitle) {
      rfs.title = true;
    }
    return rfs;
  }


  /**
   * Represents a SQuery. See {@link ForestSpec#sQuery(String, String)} for details.
   */
  @PublicApi
  public static final class SQuery {
    public static final String TYPE_JQL = "jql";
    public static final String TYPE_TEXT = "text";
    public static final String TYPE_CQL = "cql";

    @NotNull
    private final String myType;

    @NotNull
    private final String myQuery;

    /**
     * Creates SQuery instance.
     *
     * @param type one of the known query types
     * @param query query text, can be empty string
     */
    public SQuery(@NotNull String type, @NotNull String query) {
      //noinspection ConstantConditions
      if (type == null) {
        throw new IllegalArgumentException("type cannot be null");
      }
      //noinspection ConstantConditions
      if (query == null) {
        throw new IllegalArgumentException("query cannot be null");
      }
      myQuery = query;
      myType = type;
    }

    /**
     * Returns the query text.
     */
    @NotNull
    public String getQuery() {
      return myQuery;
    }

    /**
     * Returns the query type.
     */
    @NotNull
    public String getType() {
      return myType;
    }

    /**
     * Returns true if the query can be treated as "show all recent items".
     */
    @Internal
    public boolean isShowRecent() {
      return StringUtils.isEmpty(myQuery);
    }

    @Override
    public boolean equals(Object o) {
      if (this == o) return true;
      if (o == null || getClass() != o.getClass()) return false;
      SQuery sQuery = (SQuery) o;
      if (!myQuery.equals(sQuery.myQuery)) return false;
      if (!myType.equals(sQuery.myType)) return false;
      return true;
    }

    @Override
    public int hashCode() {
      int result = myQuery.hashCode();
      result = 31 * result + myType.hashCode();
      return result;
    }

    public static SQuery fromRest(RestForestSpec rfs) {
      if (rfs == null || rfs.sQuery == null) return null;
      return new SQuery(StructureUtil.nn(rfs.sQuery.type), StructureUtil.nn(rfs.sQuery.query));
    }
  }


  /**
   * Represents a transformation, which is a call to a generator with specified parameters.
   *
   * @see com.almworks.jira.structure.api.generator.CoreStructureGenerators
   * @see com.almworks.jira.structure.api.generator.CoreGeneratorParameters
   */
  public static final class Transformation {
    @NotNull
    private final String myModule;

    @Nullable
    private final Map<String, Object> myParams;

    /**
     * <p>Creates a transformation.</p>
     *
     * <p>Note that the generator, referred by {@code module}, must not be an Inserter. Although Structure
     * does not verify the type of the generator passed here, the results of passing an Inserter are not
     * defined.</p>
     *
     * @param module complete module key of the generator module
     * @param params optional map with generator parameters
     * @see com.almworks.jira.structure.api.generator.CoreStructureGenerators
     * @see com.almworks.jira.structure.api.generator.CoreGeneratorParameters
     */
    public Transformation(@NotNull String module, @Nullable Map<String, Object> params) {
      myModule = module;
      myParams = params == null ? null : JsonMapUtil.copyParameters(params, false, true, false);
    }

    /**
     * Returns the complete module key of the generator.
     */
    @NotNull
    public String getModule() {
      return myModule;
    }

    /**
     * Returns optional parameters for the generator.
     */
    @Nullable
    public Map<String, Object> getParams() {
      return myParams;
    }

    @Override
    public boolean equals(Object o) {
      if (this == o) return true;
      if (o == null || getClass() != o.getClass()) return false;
      Transformation that = (Transformation) o;
      if (!myModule.equals(that.myModule)) return false;
      if (myParams != null ? !myParams.equals(that.myParams) : that.myParams != null) return false;
      return true;
    }

    @Override
    public int hashCode() {
      int result = myModule.hashCode();
      result = 31 * result + (myParams != null ? myParams.hashCode() : 0);
      return result;
    }
  }


  /**
   * A builder for forest spec. Has setters for all forest spec properties.
   */
  public static class Builder {
    private Long myStructureId;
    private Integer myVersion;
    private SQuery mySQuery;
    private String myClipboardSessionId;
    private String myUserKey;
    private List<Transformation> myTransformations;
    private boolean mySecured;
    private boolean mySkeleton;
    private boolean myTitle;

    public Builder() {}

    /**
     * Creates a builder and copies all parameters from the given spec.
     */
    public Builder(@NotNull ForestSpec spec) {
      setStructureId(spec.getStructureId());
      setVersion(spec.getVersion());
      setSQuery(spec.getSQuery());
      setClipboardSessionId(spec.getClipboardSessionId());
      setUserKey(spec.getUserKey());
      setTransformations(spec.getTransformations());
      setSecured(spec.isSecured());
      setSkeleton(spec.isSkeleton());
      setTitle(spec.hasTitle());
    }

    @NotNull
    public Builder setStructureId(@Nullable Long structureId) {
      myStructureId = structureId;
      return this;
    }

    @NotNull
    public Builder setVersion(@Nullable Integer version) {
      myVersion = version;
      return this;
    }

    @NotNull
    public Builder setSQuery(@Nullable SQuery sQuery) {
      mySQuery = sQuery;
      return this;
    }

    @NotNull
    public Builder setSQuery(@NotNull String type, @NotNull String query) {
      return setSQuery(new SQuery(type, query));
    }

    @NotNull
    public Builder setClipboardSessionId(@Nullable String clipboardSessionId) {
      myClipboardSessionId = clipboardSessionId;
      return this;
    }

    @NotNull
    public Builder setUserKey(@Nullable String userKey) {
      myUserKey = userKey;
      return this;
    }

    @NotNull
    public Builder setCurrentUserKey() {
      myUserKey = StructureAuth.getUserKey();
      return this;
    }

    @NotNull
    public Builder setTransformations(@Nullable List<Transformation> transformations) {
      myTransformations = transformations == null || transformations.isEmpty() ? null : new ArrayList<>(transformations);
      return this;
    }

    @NotNull
    public Builder addTransformation(@NotNull Transformation transformation) {
      if (myTransformations == null) {
        myTransformations = new ArrayList<>();
      }
      myTransformations.add(transformation);
      return this;
    }

    @NotNull
    public Builder addTransformation(@NotNull String module, @Nullable Map<String, Object> params) {
      return addTransformation(new Transformation(module, params));
    }

    @NotNull
    public Builder setSecured(boolean secured) {
      mySecured = secured;
      return this;
    }

    @NotNull
    public Builder setSkeleton(boolean skeleton) {
      mySkeleton = skeleton;
      return this;
    }

    @NotNull
    public Builder setTitle(boolean title) {
      myTitle = title;
      return this;
    }

    @NotNull
    public ForestSpec build() {
      return new ForestSpec(myStructureId, myVersion, mySQuery, myClipboardSessionId, myTransformations, myUserKey, mySecured, mySkeleton, myTitle);
    }
  }


  public interface Visitor<T> {
    T visitStructure(ForestSpec structureSpec) throws StructureException;

    T visitSQuery(ForestSpec querySpec) throws StructureException;

    T visitTransformation(ForestSpec transformSpec) throws StructureException;

    T visitSecured(ForestSpec securedSpec) throws StructureException;

    T visitClipboard(ForestSpec clipboardSpec) throws StructureException;
  }
}
