package com.almworks.jira.structure.api.attribute.loader.basic;

import com.almworks.integers.IntArray;
import com.almworks.integers.IntList;
import com.almworks.jira.structure.api.attribute.*;
import com.almworks.jira.structure.api.attribute.loader.ScanningAttributeContext;
import com.almworks.jira.structure.api.attribute.loader.ScanningAttributeLoader;
import com.almworks.jira.structure.api.util.SpecParams;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Objects;
import java.util.function.Predicate;

/**
 * Base class for scanning loaders that support standard options "baseLevel" and "levels".
 *
 * Note that implementation relies on the fact that the value T is immutable and can be shared among multiple rows.
 *
 * @param <T>
 */
public abstract class AbstractScanningLoader<T> extends AbstractAttributeLoader<T> implements ScanningAttributeLoader<T> {
  private final int myBaseLevel;

  @Nullable
  private final Predicate<ScanningAttributeContext> myBaseLevelCheck;

  @Nullable
  private final IntList myLevels;

  @Nullable
  private final Predicate<ScanningAttributeContext> myLevelCheck;

  protected AbstractScanningLoader(@NotNull AttributeSpec<T> spec) {
    super(spec);
    SpecParams params = spec.getParams();

    myBaseLevel = Math.max(0, params.getInt(SharedAttributeSpecs.Param.BASE_LEVEL));
    myLevels = getLevels(params, myBaseLevel);
    myLevelCheck = myLevels == null ? null : ctx -> myLevels.contains(ctx.getDepth());
    myBaseLevelCheck = myBaseLevel == 0 ? null : ctx -> ctx.getDepth() < myBaseLevel;
  }

  protected boolean isLevelSkipped(ScanningAttributeContext context) {
    return myLevelCheck != null && !myLevelCheck.test(context);
  }

  protected boolean isAboveBaseLevel(ScanningAttributeContext context) {
    return myBaseLevelCheck != null && myBaseLevelCheck.test(context);
  }

  @Nullable
  protected Integer getBaseLevel() {
    return myBaseLevel;
  }

  @Nullable
  protected IntList getLevelsSorted() {
    return myLevels;
  }

  @Nullable
  @Override
  public AttributeValue<T> loadValue(@NotNull AttributeValue<T> precedingValue, @NotNull ScanningAttributeContext context) {
    if (isAboveBaseLevel(context)) {
      return AttributeValue.undefined();
    }
    if (isLevelSkipped(context)) {
      return carryPrecedingValueForSkippedRow(precedingValue);
    }
    if (isRowSkipped(precedingValue, context)) {
      return carryPrecedingValueForSkippedRow(precedingValue);
    }
    T value = getCarriedValue(precedingValue);
    T newValue = loadValueForPassingRow(value, context);
    if (Objects.equals(value, newValue) && precedingValue.isDefined()) {
      // reuse object
      return precedingValue;
    }
    return AttributeValue.ofNullable(newValue);
  }

  @Nullable
  protected abstract T loadValueForPassingRow(@Nullable T precedingValue, ScanningAttributeContext context);

  protected boolean isRowSkipped(AttributeValue<T> precedingValue, ScanningAttributeContext context) {
    return false;
  }

  private T getCarriedValue(AttributeValue<T> value) {
    //noinspection unchecked
    return value.isDefined() ? value.getValue() : (T) value.getLoaderData(Object.class);
  }

  protected AttributeValue<T> carryPrecedingValueForSkippedRow(AttributeValue<T> precedingValue) {
    return precedingValue.isDefined() ? AttributeValue.<T>undefined().withData(precedingValue.getValue()) : precedingValue;
  }

  @Nullable
  private static IntList getLevels(SpecParams params, int baseLevel) {
    if (!params.has(SharedAttributeSpecs.Param.LEVELS)) return null;
    IntArray levels = IntArray.create(params.getIntList(SharedAttributeSpecs.Param.LEVELS));
    for (int i = 0; i < levels.size(); i++) {
      levels.set(i, levels.get(i) - 1);
    }
    levels.sortUnique();
    // remove levels that are above base level
    int idx = levels.binarySearch(baseLevel);
    if (idx < 0) idx = -idx - 1;
    if (idx > 0) levels.removeRange(0, idx);
    return levels;
  }
}
