package com.almworks.jira.structure.api.settings;

import com.almworks.integers.*;
import com.almworks.jira.structure.api.permissions.*;
import com.almworks.jira.structure.api.structure.StructureManager;
import com.atlassian.jira.jql.builder.JqlClauseBuilder;
import com.atlassian.jira.project.Project;
import com.atlassian.jira.user.ApplicationUser;
import com.atlassian.query.Query;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.*;

import static com.almworks.jira.structure.api.permissions.CoreAppPermissions.*;
import static java.util.stream.Collectors.joining;

/**
 * <p><code>StructureConfiguration</code> provides access to the configuration parameters of the Structure app.</p>
 *
 * <p>Typically, only Jira administrators have write access to the app configuration, so before changing
 * anything you should check if the user has enough privileges - no checking is done by this component.</p>
 *
 * <p>This service also provides some utility methods that are based on the Structure configuration.</p>
 *
 * <p>All methods in this component are thread-safe.</p>
 */
public interface StructureConfiguration {
  /**
   * @return true if Structure is enabled for all projects
   * @see <a href="http://wiki.almworks.com/display/structure/Selecting+Structure-Enabled+Projects">Selecting Structure-Enabled Projects</a>
   */
  boolean isEnabledForAllProjects();

  /**
   * This method returns a list of IDs of projects picked by the administrator for the Structure app. It
   * returns a list of IDs of picked projects even if "enable for all projects" is turned on - so for the purposes
   * of checking whether Structure is available for a project, you need to use {@link #getCurrentlyEnabledProjects()}.
   *
   * @return a list of IDs of projects picked by Jira administrator on the configuration page
   * @see <a href="http://wiki.almworks.com/display/structure/Selecting+Structure-Enabled+Projects">Selecting Structure-Enabled Projects</a>
   * @since 8.2.0 (Structure 2.5)
   */
  @NotNull
  LongList getPickedProjectIds();

  /**
   * This method returns a list of projects picked by the administrator for the Structure app. It
   * returns a list of picked projects even if "enable for all projects" is turned on - so for the purposes
   * of checking whether Structure is available for a project, you need to use {@link #getCurrentlyEnabledProjects()}.
   *
   * @return a list of projects picked by Jira administrator on the configuration page
   * @see <a href="http://wiki.almworks.com/display/structure/Selecting+Structure-Enabled+Projects">Selecting Structure-Enabled Projects</a>
   */
  @NotNull
  List<Project> getPickedProjects();

  /**
   * @return a list of projects that are allowed to use Structure (if Structure is enabled for all projects,
   * returns all projects)
   * @see <a href="http://wiki.almworks.com/display/structure/Selecting+Structure-Enabled+Projects">Selecting Structure-Enabled Projects</a>
   */
  @NotNull
  List<Project> getCurrentlyEnabledProjects();

  /**
   * Checks if a given project is enabled for Structure
   *
   * @param project project to check
   * @return true if the project is not null and available for Structure
   * @see #getCurrentlyEnabledProjects()
   */
  boolean isProjectEnabled(@Nullable Project project);

  /**
   * @return true if Structure is enabled for all users (that have access to the enabled projects)
   * @see <a href="http://wiki.almworks.com/display/structure/Restricting+User+Access+to+Structure">Restricting User Access to Structure</a>
   */
  default boolean isEnabledForAnyone() {
    return isAllowedForAnyone(USE);
  }

  /**
   * Used to get the list of groups (possibly other permission subjects) that the app is enabled for.
   * Note that if the app is enabled for all users, this method would still return the groups that
   * an administrator has previously picked, but those groups are not used to check the permissions.
   *
   * @return a list of subjects that the Structure app is enabled for (when not enabled for anyone)
   * @see PermissionSubject.JiraGroup
   * @see <a href="http://wiki.almworks.com/display/structure/Restricting+User+Access+to+Structure">Restricting User Access to Structure</a>
   */
  @NotNull
  default List<PermissionSubject> getEnabledPermissionSubjects() {
    return getPermissionSubjects(USE);
  }

  /**
   * @return true if creating new structures is enabled for all users (that have access to Structure at all)
   * @see <a href="http://wiki.almworks.com/display/structure/Changing+Permission+to+Create+New+Structures">Changing Permission to Create New Structures</a>
   */
  default boolean isCreateEnabledForAnyone() {
    return isAllowedForAnyone(CREATE_STRUCTURE);
  }

  /**
   * Used to get the list of groups (possibly other permission subjects) that are allowed to create
   * new structures.
   * Note that if structure creation is enabled for all users, this method would still return the groups that
   * an administrator has previously picked, but those groups are not used to check the permissions.
   *
   * @return a list of subjects that are allowed to create new structures (when not enabled for anyone)
   * @see PermissionSubject.JiraGroup
   * @see <a href="http://wiki.almworks.com/display/structure/Changing+Permission+to+Create+New+Structures">Changing Permission to Create New Structures</a>
   */
  @NotNull
  default List<PermissionSubject> getCreatorPermissionSubjects() {
    return getPermissionSubjects(CREATE_STRUCTURE);
  }

  /**
   * @return true if synchronization is available for all users who have control access to the structure
   * @since 8.5.0 (Structure 2.8)
   */
  default boolean isSynchronizationEnabledForAnyone() {
    return isAllowedForAnyone(SYNCHRONIZATION);
  }

  /**
   * Used to get the list of groups (possibly other permission subjects) that are allowed to configure and control synchronizers of controlled structures.
   * @return a list of subjects that are allowed to configure and control synchronizers (when not enabled for anyone)
   * @since 8.5.0 (Structure 2.8)
   */
  @NotNull
  default List<PermissionSubject> getSynchronizationPermissionSubjects() {
    return getPermissionSubjects(SYNCHRONIZATION);
  }

  /**
   * @return true if automation is available for all users who have control access to the structure
   * @since 10.0.0 (Structure 3.0)
   */
  default boolean isAutomationEnabledForAnyone() {
    return isAllowedForAnyone(AUTOMATION);
  }

  /**
   * Used to get the list of groups (possibly other permission subjects) that are allowed to configure and control automation of controlled structures.
   * @return a list of subjects that are allowed to configure and control automation (when not enabled for anyone)
   * @since 10.0.0 (Structure 3.0)
   */
  @NotNull
  default List<PermissionSubject> getAutomationPermissionSubjects() {
    return getPermissionSubjects(AUTOMATION);
  }

  /**
   * Changes whether the app is enabled for all projects.
   *
   * @param enabled if true, Structure will be enabled for all projects
   * @see <a href="http://wiki.almworks.com/display/structure/Selecting+Structure-Enabled+Projects">Selecting Structure-Enabled Projects</a>
   */
  void setEnabledForAllProjects(boolean enabled);

  /**
   * Changes projects that Structure is enabled for. The setting is effective only if
   * {@link #isEnabledForAllProjects} returns false.
   *
   * @param idList a comma-delimited list of project IDs
   * @see <a href="http://wiki.almworks.com/display/structure/Selecting+Structure-Enabled+Projects">Selecting Structure-Enabled Projects</a>
   */
  void setPickedProjectIds(@Nullable String idList);

  /**
   * Changes projects that Structure is enabled for. The setting is effective only if
   * {@link #isEnabledForAllProjects} returns false.
   *
   * @param projectIds collection of project IDs
   * @see <a href="http://wiki.almworks.com/display/structure/Selecting+Structure-Enabled+Projects">Selecting Structure-Enabled Projects</a>
   */
  default void setPickedProjectIds(@Nullable LongIterable projectIds) {
    if (projectIds == null) {
      setPickedProjectIds("");
      return;
    }
    StringJoiner joiner = new StringJoiner(",");
    for (LongIterator it : projectIds) {
      joiner.add(Long.toString(it.value()));
    }
    setPickedProjectIds(joiner.toString());
  }

  /**
   * Changes projects that Structure is enabled for. The setting is effective only if
   * {@link #isEnabledForAllProjects} returns false.
   *
   * @param projectIds collection of project IDs
   * @see <a href="http://wiki.almworks.com/display/structure/Selecting+Structure-Enabled+Projects">Selecting Structure-Enabled Projects</a>
   */
  default void setPickedProjectIds(@Nullable Collection<Long> projectIds) {
    setPickedProjectIds(projectIds == null ? "" :
      projectIds.stream().map(String::valueOf).collect(joining(",")));
  }

  /**
   * Changes whether the app is available for all users.
   *
   * @param enabled if true, Structure will be available to all users that have access to at least one structure-enabled project.
   * @see <a href="http://wiki.almworks.com/display/structure/Restricting+User+Access+to+Structure">Restricting User Access to Structure</a>
   */
  default void setEnabledForAnyone(boolean enabled) {
    setAllowedForAnyone(USE, enabled);
  }

  /**
   * Changes the list of the users / groups that the Structure app is enabled for. When Structure
   * is configured to be enabled for anyone, this method has no effect.
   *
   * @param subjects comma-delimited list of string-encoded permission subjects
   * @see PermissionSubject#toEncodedString()
   * @see <a href="http://wiki.almworks.com/display/structure/Restricting+User+Access+to+Structure">Restricting User Access to Structure</a>
   */
  default void setEnabledPermissionSubjectsEncoded(@Nullable String subjects) {
    setPermissionSubjectsEncoded(USE, subjects);
  }

  /**
   * Changes whether the creation of new structures is available for all users.
   *
   * @param enabled if true, anyone who has access to Structure can create new structures.
   * @see <a href="http://wiki.almworks.com/display/structure/Changing+Permission+to+Create+New+Structures">Changing Permission to Create New Structures</a>
   */
  default void setCreateEnabledForAnyone(boolean enabled) {
    setAllowedForAnyone(CREATE_STRUCTURE, enabled);
  }

  /**
   * Changes the list of the users / groups that are allowed to create new structures. When Structure is configured
   * to allow anyone to create new structures, this setting has no effect.
   *
   * @param subjects comma-delimited list of string-encoded permission subjects
   * @see PermissionSubject#toEncodedString()
   * @see <a href="http://wiki.almworks.com/display/structure/Changing+Permission+to+Create+New+Structures">Changing Permission to Create New Structures</a>
   */
  default void setCreatorPermissionSubjectsEncoded(@Nullable String subjects) {
    setPermissionSubjectsEncoded(CREATE_STRUCTURE, subjects);
  }

  /**
   * Changes whether the synchronization is available for all users.
   *
   * @param enabled if true, anyone who has control access to the structure can configure and control synchronizers.
   * @since 8.5.0 (Structure 2.8)
   */
  default void setSynchronizationEnabledForAnyone(boolean enabled) {
    setAllowedForAnyone(SYNCHRONIZATION, enabled);
  }

  /**
   * Changes the list of the users / groups that are allowed to configure and control synchronizers in controlled structures.
   *
   * @param subjects comma-delimited list of string-encoded permission subjects
   * @since 8.5.0 (Structure 2.8)
   */
  default void setSynchronizationPermissionSubjectsEncoded(@Nullable String subjects) {
    setPermissionSubjectsEncoded(SYNCHRONIZATION, subjects);
  }

  /**
   * Changes whether the automation is available for all users.
   *
   * @param enabled if true, anyone who has control access to the structure can configure and control automation.
   * @since 10.0.0 (Structure 3.0)
   */
  default void setAutomationEnabledForAnyone(boolean enabled) {
    setAllowedForAnyone(AUTOMATION, enabled);
  }

  /**
   * Changes the list of the users / groups that are allowed to configure and control automation in controlled structures.
   *
   * @param subjects comma-delimited list of string-encoded permission subjects
   * @since 10.0.0 (Structure 3.0)
   */
  default void setAutomationPermissionSubjectsEncoded(@Nullable String subjects) {
    setPermissionSubjectsEncoded(AUTOMATION, subjects);
  }

  /**
   * <p>Adds to the JQL builder a condition that limits the result set to the projects enabled for Structure.</p>
   * <p>This method adds JQL constraint <code>project IN (....)</code> in case Structure is allowed only for
   * specific projects and does nothing if Structure is allowed for all projects. The condition is added at the
   * current level in the clause builder.</p>
   *
   * @param builder a clause builder to be modified; if null, a new clause builder is created
   * @return the builder that was passed as the parameter or the newly created builder
   */
  @NotNull
  JqlClauseBuilder addConfigurationScopeClause(@Nullable JqlClauseBuilder builder);

  /**
   * Utility method that returns a JQL query that selects all issues in the projects that are available for Structure.
   * This is a convenience method that uses {@link #addConfigurationScopeClause}.
   *
   * @return a query that selects issues enabled for Structure, or null if all issues are enabled
   */
  @Nullable
  Query getConfigurationScopeQuery();

  /**
   * <p>Checks if Structure is available for the specified user.
   *
   * <p>This is not the same as {@code isAllowed(CoreAppPermissions.USE, user)},
   * because it also checks user's access to enabled projects.
   *
   * @param user the user, null means anonymous
   * @return true if the user can use Structure
   * @see <a href="http://wiki.almworks.com/display/structure/Who+Has+Access+to+the+Structure">Who Has Access to the Structure</a>
   */
  boolean isStructureAvailable(@Nullable ApplicationUser user);

  /**
   * Checks if the user is allowed to create new structures.
   *
   * @param user the user, null means anonymous
   * @return true if the user can create new structures
   * @see <a href="http://wiki.almworks.com/display/structure/Who+Has+Access+to+the+Structure">Who Has Access to the Structure</a>
   */
  default boolean isStructureCreationAllowed(@Nullable ApplicationUser user) {
    return isAllowed(CREATE_STRUCTURE, user);
  }

  /**
   * Checks if the user is allowed to configure and control synchronizers.
   *
   * @param user the user, null means anonymous
   * @return true if the user can configure and control synchronizers
   * @since 8.5.0 (Structure 2.8)
   */
  default boolean isSynchronizationAllowed(@Nullable ApplicationUser user) {
    return isAllowed(SYNCHRONIZATION, user);
  }

  /**
   * Checks if the user is allowed to configure and use automation.
   *
   * @param user the user, null means anonymous
   * @return true if the user can configure and control automation
   * @since 10.0.0 (Structure 3.0)
   */
  default boolean isAutomationAccessAllowed(@Nullable ApplicationUser user){
    return isAllowed(AUTOMATION, user);
  }

  /**
   * Checks if the given Structure app permission is granted to all users.
   *
   * @param permission the permission
   * @return true if the permission is granted to all users
   * @see CoreAppPermissions
   */
  boolean isAllowedForAnyone(@NotNull StructureAppPermission permission);

  /**
   * Returns the list of groups (possibly other permission subjects) that are
   * granted the given Structure app permission.
   *
   * @param permission the permission
   * @return the list of subjects that are granted the permission
   * @see CoreAppPermissions
   */
  @NotNull
  List<PermissionSubject> getPermissionSubjects(@NotNull StructureAppPermission permission);

  /**
   * Grant the given Structure app permission to anyone.
   *
   * @param permission the permission
   * @param allowed true to grant the permission to anyone
   * @see CoreAppPermissions
   */
  void setAllowedForAnyone(@NotNull StructureAppPermission permission, boolean allowed);

  /**
   * Changes the list of the users / groups that are granted the given
   * Structure app permission.
   *
   * @param permission the permission
   * @param subjects comma-delimited list of string-encoded permission subjects
   * @see CoreAppPermissions
   */
  void setPermissionSubjectsEncoded(@NotNull StructureAppPermission permission, @Nullable String subjects);

  /**
   * Changes the list of the users / groups that are granted the given
   * Structure app permission.
   *
   * @param permission the permission
   * @param subjects the collection of permission subjects
   * @see CoreAppPermissions
   */
  default void setPermissionSubjects(@NotNull StructureAppPermission permission, @Nullable Collection<? extends PermissionSubject> subjects) {
    setPermissionSubjectsEncoded(permission, subjects == null ? "" :
      subjects.stream().map(PermissionSubject::toEncodedString).collect(joining(",")));
  }

  /**
   * Checks if the user is granted the given Structure app permission.
   *
   * @param permission the permission
   * @param user the user, null means anonymous
   * @return true if the user is granted the permission
   * @see CoreAppPermissions
   */
  boolean isAllowed(@NotNull StructureAppPermission permission, @Nullable ApplicationUser user);

  /**
   * <p>Returns the ID of the default structure for a given project. If <code>project</code> is <code>null</code>, return system-wide
   * default structure.</p>
   * <p>Returns 0 if there's no default structure available.</p>
   * <p>The Structure app stores a system default structure. It also allows project administrator to change default structure for a project.</p>
   * <p>Structure user interface can be configured to switch to the project default structure when that project
   * or issue from that project is opened.</p>
   * <p>
   *   <strong>Important:</strong> It is not guaranteed that the structure with this given ID exists! The preference
   *   is stored independently from the structures.
   * </p>
   *
   * @param project the project for which default structure is requested; <code>null</code> to get a system-wide default structure
   * @return the ID of the default structure for the specified project, or 0
   * @see StructureManager#getStructure(Long, PermissionLevel)
   * @see #setDefaultStructureId(com.atlassian.jira.project.Project, Long)
   */
  long getDefaultStructureId(@Nullable Project project);

  /**
   * <p>Changes the ID of the default structure for the specified project. If <code>project</code> is <code>null</code>,
   * updates system-wide default structure.</p>
   * <p>See {@link #getDefaultStructureId(com.atlassian.jira.project.Project)} for more details about default structures.</p>
   *
   * @param project the project for which to set the default structure, <code>null</code> to set system-wide default
   * @param structureId the ID of the default structure, <code>null</code> to clear the value (project-level default structure
   * will use system-level default structure, and system-level default structure ID will be reset to 0 - no structure)
   * @see #getDefaultStructureId(com.atlassian.jira.project.Project)
   */
  void setDefaultStructureId(@Nullable Project project, @Nullable Long structureId);

  /**
   * Used to check whether a specific project has system-level default structure overridden with project-level default structure.
   *
   * @param project project to check
   * @return true if a project-level default structure has been specified
   * @see #setDefaultStructureId(com.atlassian.jira.project.Project, Long)
   */
  boolean isDefaultStructureSetForProject(@NotNull Project project);

  /**
   * <p>Retrieves user interface settings for the specified user and project.</p>
   *
   * <p>UI settings are typically defined at a system level by Jira administrator
   * and can be overridden by each user and for each project. This method calculates the final UI settings
   * given which user is viewing Structure widget and the project context (such as viewing an issue or a project tab).</p>
   *
   * <p>
   *   The following is the order in which settings are applied:
   * </p>
   * <ol>
   *   <li>Per-user per-project value, if set</li>
   *   <li>Per-user value, if set</li>
   *   <li>Per-project value, if set</li>
   *   <li>System default value</li>
   * </ol>
   *
   * @param user the user to retrieve UI settings for, or <code>null</code> for anonymous user
   * @param project the project to retrieve UI settings for, or <code>null</code> for non-project-specific settings
   * @return an instance of {@link UISettings} that details how Structure Widget should work
   */
  @NotNull
  UISettings getUISettings(@Nullable ApplicationUser user, @Nullable Project project);

  /**
   * <p>This method is used to update the user interface settings - either system-wide default
   * settings or per-user settings or per-project settings. See {@link #getUISettings(ApplicationUser, Project)}
   * for details.</p>
   *
   * <p><strong>Per-project UI settings are not yet supported.</strong>
   * Using <code>project</code> argument currently has no effect -
   * it would work as if <code>project</code> was <code>null</code>. The argument is added for
   * the sake of forward compatibility as we expect to support it soon.</p>
   *
   * <p>Use {@link com.almworks.jira.structure.api.settings.UISettingsBean} to
   * specify the settings you want to change. If a certain setting has <code>null</code>
   * value, it is treated as "not set" and the value is inherited from generic settings.</p>
   *
   * <p>To unset a specific setting and make it inherit a default value, use {@link #clearUISettings}
   * to clear all settings and then this method to restore all overridden settings you'd like to keep.</p>
   *
   * @param user the user to apply the settings for, or <code>null</code> to apply globally
   * @param project (currently not used) the project to apply settings for, or <code>null</code> to apply across all projects
   * @param settings an instance of settings
   */
  void setUISettings(@Nullable ApplicationUser user, @Nullable Project project, @NotNull UISettings settings);

  /**
   * <p>Completely removes per-user or per-project settings. Next time settings are retrieved,
   * default/inherited settings will take effect.</p>
   *
   * <p>Either <code>user</code> or <code>project</code> settings must be specified. This method
   * has no effect on the system defaults.</p>
   *
   * @param user the user to clear settings for
   * @param project the project to clear settings for
   * @see #setUISettings
   * @see UISettings
   */
  void clearUISettings(@Nullable ApplicationUser user, @Nullable Project project);

  @NotNull
  AttributeSensitivitySettings getAttributeSensitivitySettings();

  void setAttributeSensitivitySettings(@NotNull AttributeSensitivitySettings settings);
}
