package com.almworks.jira.structure.api.effect;

import com.almworks.jira.structure.api.util.JsonMapUtil;
import com.atlassian.annotations.PublicApi;
import org.codehaus.jackson.annotate.JsonCreator;
import org.codehaus.jackson.annotate.JsonProperty;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.*;

/**
 * A description of an effect that can be stored in the database or passed
 * over the wire. Contains an {@link EffectProvider effect provider} module key
 * and implementation-specific parameters.
 */
@PublicApi
public final class StoredEffect {
  @NotNull
  private final String myModuleKey;

  @NotNull
  private final Map<String, Object> myParameters;

  @JsonCreator
  public StoredEffect(
    @JsonProperty("moduleKey") @NotNull String moduleKey,
    @JsonProperty("parameters") @NotNull Map<String, Object> parameters)
  {
    myModuleKey = moduleKey;
    myParameters = JsonMapUtil.copyParameters(parameters, false, true, true);
  }

  /**
   * The complete module key of the {@link EffectProvider} responsible
   * for converting this effect description to an actual {@link Effect}.
   */
  @NotNull
  public String getModuleKey() {
    return myModuleKey;
  }

  /**
   * Provider-specific parameters, e.g. an issue ID, a field ID, and a value.
   */
  @NotNull
  public Map<String, Object> getParameters() {
    return myParameters;
  }

  /**
   * Returns a new builder with the given module key.
   */
  @NotNull
  public static Builder builder(@NotNull String moduleKey) {
    return new Builder(moduleKey);
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;
    StoredEffect effect = (StoredEffect) o;
    return myModuleKey.equals(effect.myModuleKey) &&
      myParameters.equals(effect.myParameters);
  }

  @Override
  public String toString() {
    return "StoredEffect{" +
      "myModuleKey='" + myModuleKey + '\'' +
      ", myParameters=" + myParameters +
      '}';
  }

  @Override
  public int hashCode() {
    return Objects.hash(myModuleKey, myParameters);
  }


  /**
   * Builder class for {@link StoredEffect effect descriptions}.
   */
  public static final class Builder {
    @NotNull
    private String myModuleKey;

    @NotNull
    private final Map<String, Object> myParameters = new HashMap<>();

    /**
     * Creates a new builder with the given effect provider module key
     * and no parameters.
     *
     * @param moduleKey the module key
     */
    public Builder(@NotNull String moduleKey) {
      myModuleKey = moduleKey;
    }

    /**
     * Creates a new builder with the module key and parameters taken
     * from the given effect description.
     *
     * @param storedEffect the source effect description
     */
    public Builder(@NotNull StoredEffect storedEffect) {
      myModuleKey = storedEffect.getModuleKey();
      myParameters.putAll(storedEffect.getParameters());
    }

    /**
     * Updates the effect provider module key.
     *
     * @param moduleKey the module key
     * @return this builder
     */
    @NotNull
    public Builder setModuleKey(@NotNull String moduleKey) {
      myModuleKey = moduleKey;
      return this;
    }

    /**
     * Updates a provider-specific parameter.
     *
     * @param key parameter key
     * @param value parameter value; passing {@code null} deletes the parameter
     * @return this builder
     */
    @NotNull
    public Builder setParameter(@NotNull String key, @Nullable Object value) {
      if (value != null) {
        myParameters.put(key, value);
      } else {
        myParameters.remove(key);
      }
      return this;
    }

    /**
     * Creates and returns a new {@link StoredEffect} instance with the
     * module key and parameters from this builder.
     *
     * @return a new effect description
     */
    @NotNull
    public StoredEffect build() {
      return new StoredEffect(myModuleKey, myParameters);
    }
  }
}
