package com.almworks.jira.structure.api.backup;

import com.almworks.integers.LongIterable;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.io.File;
import java.io.OutputStream;
import java.util.Set;

/**
 * <p>Backup operation allows to set backup parameters and run the backup.</p>
 *
 * <p>Note that for security reasons, if the file name is entered by the user, you must
 * check that it is located under the JIRA home directory, or force the file to be under that
 * directory. This operation trusts the caller and doesn't do any checking.</p>
 *
 * <p>If the specified backup file already exists, it will be overwritten.</p>
 */
public interface BackupOperation {
  /**
   * <p>Sets the target file for the backup. If the filename does not have extension,
   * ".xml" or ".zip" will be appended automatically, depending on the <code>useZip</code> flag.</p>
   *
   * <p>The file must be an absolute path.</p>
   *
   * @param file backup file
   * @return this operation
   * @throws IllegalArgumentException if the file is not a valid path
   */
  @NotNull
  BackupOperation setFile(@NotNull File file);

  /**
   * <p>Sets the <code>useZip</code> flag. When the flag is set, the target file will be a zipped
   * XML file, otherwise it will be a plain XML file.
   *
   * <p>The default is {@code false}.
   *
   * @param useZip true if the file content should be compressed
   * @return this operation
   */
  @NotNull
  BackupOperation setUseZip(boolean useZip);

  /**
   * <p>Sets the structure IDs to back up. If set to {@code null}, all structures
   * will be backed up. If set to an empty collection, no structures will be backed up.
   *
   * <p>The default is {@code null} (all structures).
   *
   * @param structureIds the collection of structure IDs; {@code null} means all structures
   * @return this operation
   */
  @NotNull
  BackupOperation setBackupStructureIds(@Nullable LongIterable structureIds);

  /**
   * <p>Sets the view IDs to back up. If set to {@code null}, all views
   * will be backed up. If set to an empty collection, no views will be backed up.
   *
   * <p>The default is {@code null} (all views).
   *
   * @param viewIds the collection of view IDs; {@code null} means all views
   * @return this operation
   */
  @NotNull
  BackupOperation setBackupViewIds(@Nullable LongIterable viewIds);

  /**
   * <p>Controls whether Structure's app-level configuration and settings are
   * backed up. This includes:
   * <ul>
   *   <li>the list of projects enabled for Structure;
   *   <li>Structure app permissions (who can use the app, create new structures, etc.);
   *   <li>attribute sensitivity settings;
   *   <li>scheduled maintenance settings;
   *   <li>dark features, and several other settings.
   * </ul>
   *
   * <p>The default is {@code true}.
   *
   * @param backupAppConfiguration {@code true} to back up Structure app configuration
   * and global permissions; {@code false} to skip them
   * @return this operation
   */
  @NotNull
  BackupOperation setBackupAppConfiguration(boolean backupAppConfiguration);

  /**
   * <p>Controls whether perspectives are backed up.
   *
   * <p>The default is {@code true}.
   *
   * @param backupPerspectives {@code true} to back up perspectives, {@code false} to skip them
   * @return this operation
   */
  @NotNull
  BackupOperation setBackupPerspectives(boolean backupPerspectives);

  /**
   * <p>Controls whether users' favorite structures are backed up.
   *
   * <p>The default is {@code true}.
   *
   * @param backupFavorites {@code true} to back up favorites, {@code false} to skip them
   * @return this operation
   */
  @NotNull
  BackupOperation setBackupFavorites(boolean backupFavorites);

  /**
   * <p>Controls whether saved columns are backed up.
   *
   * <p>The default is {@code true}.
   *
   * @param backupSavedColumns {@code true} to back up saved columns, {@code false} to skip them
   * @return this operation
   */
  BackupOperation setBackupSavedColumns(boolean backupSavedColumns);

  /**
   * <p>Sets the <code>backupHistory</code> flag. When the flag is set, full structure history
   * will be backed up, otherwise the target file will only contain the current state
   * of the structures.
   *
   * <p>The default is {@code false}.
   *
   * @param backupHistory true if structure history should be backed up
   * @return this operation
   */
  @NotNull
  BackupOperation setBackupHistory(boolean backupHistory);

  /**
   * <p>Sets the extension keys to back up. If set to {@code null}, all supported extensions
   * will be backed up. If set to an empty set, no extensions will be backed up.
   *
   * <p>The default is {@code null} (all supported extensions).
   *
   * @param extensionKeys the set of extension keys; {@code null} means all supported extensions
   * @return this operation
   */
  @NotNull
  BackupOperation setBackupExtensions(@Nullable Set<String> extensionKeys);

  /**
   * Performs the backup synchronously.
   *
   * @return this operation
   * @throws Exception if file cannot be written or any other problem happens
   */
  @NotNull
  BackupOperation backup() throws Exception;

  /**
   * <p>Performs the backup synchronously, writing the data to the supplied
   * {@link OutputStream} instead of a file. Does not close the stream.
   *
   * <p>If the {@link #setUseZip(boolean) useZip} flag was set, the data will be zipped,
   * otherwise it will be plain XML.
   *
   * <p>When backing up to a stream, the file path supplied to {@link #setFile(File)}
   * is ignored, and {@link #getFinalFile()} will always throw an {@link IllegalStateException}.
   *
   * @param out the output stream
   * @return this operation
   * @throws Exception if data cannot be written or any other problem happens
   */
  @NotNull
  BackupOperation backupToStream(@NotNull OutputStream out) throws Exception;

  /**
   * @return the file name that will be used for backup - that was set with {@link #setFile} method, but probably
   * with added extension.
   *
   * @throws IllegalStateException if the file has not been set
   */
  @NotNull
  File getFinalFile();
}
