package com.almworks.jira.structure.api.attribute.loader.basic;

import com.almworks.jira.structure.api.attribute.AttributeSpec;
import com.almworks.jira.structure.api.attribute.AttributeValue;
import com.almworks.jira.structure.api.attribute.loader.AttributeLoader;
import com.almworks.jira.structure.api.attribute.loader.DerivedAttributeContext;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collections;
import java.util.Set;

public abstract class SimpleDerivedAttributeLoader<T, D> extends AbstractDerivedAttributeLoader<T> {
  protected final AttributeSpec<? extends D> myDependency;
  private final boolean myYield;

  protected SimpleDerivedAttributeLoader(AttributeSpec<T> spec, AttributeSpec<? extends D> dependency) {
    this(spec, dependency, true);
  }

  /**
   * @param spec
   * @param dependency
   * @param yield if true, then error and null values will be returned as null, thus allowing other loaders
   * to provide other values for the attribute; if false, this method will always return something
   */
  protected SimpleDerivedAttributeLoader(AttributeSpec<T> spec, AttributeSpec<? extends D> dependency, boolean yield) {
    super(spec);
    myDependency = dependency;
    myYield = yield;
  }

  public final Set<AttributeSpec<?>> getAttributeDependencies() {
    return Collections.singleton(myDependency);
  }

  @Override
  public AttributeValue<T> loadValue(DerivedAttributeContext context) {
    AttributeValue<? extends D> dependentResult = context.getDependencyAttributeValue(myDependency);
    D dependentValue = dependentResult.getValue();
    AttributeValue<T> result;
    if (dependentValue == null) {
      // keeps loader data and error flag
      result = myYield ? null : dependentResult.cast(getAttributeSpec());
    } else {
      T value = getValue(dependentValue, context);
      result = value == null && myYield ? null : AttributeValue.ofNullable(value);
    }
    return result;
  }

  /**
   *
   * @param value
   * @param context
   * @return null if the result should be undefined (or yielded to other loaders)
   */
  @Nullable
  protected abstract T getValue(@NotNull D value, DerivedAttributeContext context);

  public static <T> AttributeLoader<T> idLoader(AttributeSpec<T> spec, AttributeSpec<T> dependency) {
    if (spec.getFormat() != dependency.getFormat()) return null;
    return new SimpleDerivedAttributeLoader<T, T>(spec, dependency) {
      @Override
      protected T getValue(@NotNull T value, DerivedAttributeContext context) {
        return value;
      }
    };
  }
}
