package com.almworks.jira.structure.api.item.generic;

import com.almworks.integers.LongIterator;
import com.almworks.jira.structure.api.StructurePluginHelper;
import com.almworks.jira.structure.api.auth.StructureAuth;
import com.almworks.jira.structure.api.error.StructureException;
import com.almworks.jira.structure.api.item.*;
import com.almworks.jira.structure.api.row.DummyRow;
import com.almworks.jira.structure.api.row.StructureRow;
import com.almworks.jira.structure.api.util.StructureUtil;
import com.atlassian.annotations.Internal;
import com.atlassian.annotations.PublicApi;
import com.atlassian.jira.user.ApplicationUser;
import com.atlassian.jira.util.ErrorCollection;
import com.atlassian.plugin.ModuleDescriptor;
import org.apache.commons.lang.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Map;

import static com.almworks.jira.structure.api.error.StructureErrors.INVALID_PARAMETER;
import static com.almworks.jira.structure.api.error.StructureErrors.ITEM_NOT_EXISTS_OR_NOT_ACCESSIBLE;

/**
 * <p>{@code GenericItemType} is responsible for generic items support in Structure plugin.</p>
 *
 * <p>It is used as a reference in structure-item-type module that declares generic item type.</p>
 */
@PublicApi
public class GenericItemType implements StructureItemType<GenericItem>, CreatableItemType, UpdatableItemType, StructureOwnedItemType, DeadItemsCheckingItemType, BulkAccessCheckingItemType {
  public static final long NEW_ITEM_ID = Long.MAX_VALUE;

  private final GenericItemService myGenericItemService;
  private final GenericItemManager myGenericItemManager;
  private final StructurePluginHelper myHelper;
  private volatile String myItemType;

  public GenericItemType(GenericItemService genericItemService, GenericItemManager genericItemManager, StructurePluginHelper helper) {
    myGenericItemService = genericItemService;
    myGenericItemManager = genericItemManager;
    myHelper = helper;
  }

  @Internal
  public void init(ModuleDescriptor descriptor) {
    if (myItemType == null) {
      myItemType = descriptor.getCompleteKey();
    }
  }

  @Nullable
  @Override
  public GenericItem accessItem(@NotNull ItemIdentity itemId) {
    if (myItemType.equals(itemId.getItemType()) && itemId.isLongId() && itemId.getLongId() != NEW_ITEM_ID) {
      return myGenericItemManager.getItem(itemId);
    }
    return null;
  }

  @Override
  public boolean isVisible(@NotNull GenericItem item, @Nullable ApplicationUser user) {
    return myGenericItemManager.isVisible(item, user);
  }

  @Override
  public StructureRow createDummyRow(long rowId, long semantics, Map<String, Object> values,
    ErrorCollection errors) throws StructureException
  {
    String name = getName(values, errors);
    if (name == null) {
      return null;
    }
    GenericItem genericItem = GenericItem.named(name).build();
    GenericItemService.CreateValidationResult validationResult = myGenericItemService.validateCreate(myItemType, genericItem);
    if (validationResult.isValid()) {
      return new DummyRow(rowId, ItemIdentity.longId(myItemType, NEW_ITEM_ID), semantics, genericItem);
    } else {
      handleInvalidResult(validationResult, errors);
      return null;
    }
  }

  @Override
  public ItemIdentity createItem(Map<String, Object> values, ErrorCollection errors) throws StructureException {
    String name = getName(values, errors);
    if (name == null) {
      return null;
    }
    GenericItem.Builder builder = GenericItem.named(name);
    readItemValues(builder, values, null, errors);
    if (errors.hasAnyErrors()) return null;
    GenericItem genericItem = builder.build();
    GenericItemService.CreateValidationResult validationResult = myGenericItemService.validateCreate(myItemType, genericItem);
    GenericItemService.GenericItemResult createResult = myGenericItemService.create(validationResult);
    if (createResult.isValid()) {
      return createResult.getItemId();
    } else {
      handleInvalidResult(createResult, errors);
      return null;
    }
  }

  @Override
  public void updateItem(ItemIdentity itemId, Map<String, Object> values,
    ErrorCollection errors) throws StructureException
  {
    if (!myItemType.equals(itemId.getItemType())) {
      throw INVALID_PARAMETER.withMessage(itemId + " is not a " + myItemType + " ID");
    }
    GenericItem existing = accessItem(itemId);
    if (existing == null || !isVisible(existing, StructureAuth.getUser())) {
      throw ITEM_NOT_EXISTS_OR_NOT_ACCESSIBLE.forItem(itemId).withMessage(itemId + " is not found or not accessible");
    }
    GenericItem.Builder builder = GenericItem.copy(existing);
    if (values.containsKey("summary")) {
      String name = getName(values, errors);
      if (name == null) {
        return;
      }
      builder.setName(name);
    }
    readItemValues(builder, values, existing, errors);
    if (errors.hasAnyErrors()) return;
    GenericItem genericItem = builder.build();
    GenericItemService.UpdateValidationResult validationResult = myGenericItemService.validateUpdate(itemId, genericItem);
    GenericItemService.GenericItemResult updateResult = myGenericItemService.update(validationResult);
    if (!updateResult.isValid()) {
      handleInvalidResult(updateResult, errors);
    }
  }

  @NotNull
  @Override
  public ItemIdentity getOwnedItem(@NotNull ItemIdentity itemId, long structureId,
    boolean copyIfSame) throws StructureException
  {
    if (myItemType.equals(itemId.getItemType()) && itemId.isLongId()) {
      return myGenericItemManager.getOwnedItem(itemId, structureId, copyIfSame);
    }
    return itemId;
  }

  @Override
  public void filterDead(ItemIdentitySet items, ItemIdentitySet deadCollector) {
    for (LongIterator it : items.longIds(myItemType)) {
      ItemIdentity itemId = ItemIdentity.longId(myItemType, it.value());
      GenericItem genericItem = accessItem(itemId);
      if (genericItem == null) {
        deadCollector.add(itemId);
      }
    }
  }

  @Override
  public void filterInaccessible(ItemIdentitySet items, ApplicationUser user, boolean overrideSecurity,
    ItemIdentitySet inaccessibleCollector)
  {
    for (LongIterator it : items.longIds(myItemType)) {
      if (it.value() == NEW_ITEM_ID) {
        continue;
      }
      ItemIdentity itemId = ItemIdentity.longId(myItemType, it.value());
      GenericItem item = accessItem(itemId);
      if (item == null || (!overrideSecurity && !isVisible(item, user))) {
        inaccessibleCollector.add(itemId);
      }
    }
  }

  @Nullable
  private String getName(Map<String, Object> values, ErrorCollection errors) {
    String name = StructureUtil.getSingleParameter(values, "summary");
    if (StringUtils.isBlank(name)) {
      errors.addError("summary", myHelper.getI18n().getText("s.ext.it.item.error.name-blank"));
      return null;
    }
    return name;
  }

  protected void readItemValues(GenericItem.Builder builder, Map<String, Object> values, GenericItem existing, ErrorCollection errors) {
    if (values == null) {
      return;
    }
    if (values.containsKey("description")) {
      builder.setDescription(StructureUtil.getSingleParameter(values, "description"));
    }
    if (values.containsKey("params")) {
      Map<String, Object> params = StructureUtil.toMap(values.get("params"));
      if (params == null) {
        builder.setParameters(null);
      } else {
        params.forEach(builder::setParameter);
      }
    }
  }

  private void handleInvalidResult(GenericItemService.ExceptionSupportResult result,
    ErrorCollection errors) throws StructureException
  {
    if (result.getException() != null) {
      throw result.getException();
    }
    errors.addErrorCollection(result.getErrorCollection());
  }
}