package com.almworks.jira.structure.api.forest;

import com.almworks.jira.structure.api.error.StructureException;
import com.atlassian.annotations.PublicApi;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

/**
 * <p>{@code ForestService} manages all structured content.</p>
 *
 * <p>This component provides instances of {@link ForestSource} that correspond to a {@link ForestSpec}. The
 * forest spec serves as an identifier of the forest and a recipe for its assembly. The forest source provides
 * live updates to the forest content that can be shown to the user or used in calculations.</p>
 *
 * <p>The automation engine, which produces dynamic content for the forests, is also working behind the scenes
 * in this component. Unless {@code skeleton} flag is specified in the {@code ForestSpec} (and also unless showing
 * a historical version of a structure), all content coming out of forest source will have the results of automation.</p>
 *
 * <p>The instances of {@code ForestSource} can be cached by Structure and shared among different threads and callers.
 * However, the caller should not store {@code ForestSource} for an extended period of time &mdash; normally,
 * an instance must be requested from {@code ForestService} every time a new request comes.</p>
 *
 * @see ForestSpec
 * @see ForestSource
 */
@PublicApi
public interface ForestService {
  /**
   * <p>Creates a forest source based on the forest spec.</p>
   *
   * <p>If {@code spec} is secured (see {@link ForestSpec#secure(String)}, this method also checks that
   * the user the {@code spec} is secured for has permissions to view the referred structure. If permission
   * check fails, {@code StructureException} is thrown.</p>
   *
   * @param spec forest specification
   * @return forest source
   * @throws StructureException if the forest source cannot be created
   */
  @NotNull
  ForestSource getForestSource(@Nullable ForestSpec spec) throws StructureException;

  /**
   * <p>Suggests a user-readable name fo the given forest spec. This can be displayed as the title of the forest.</p>
   *
   * <p>Provides text in the current user's locale.</p>
   *
   * @param spec forest specification
   * @return non-empty string that can be used to name the content
   */
  @NotNull
  String getDisplayName(@Nullable ForestSpec spec);
}
