package com.almworks.jira.structure.api.attribute.subscription;

import com.almworks.integers.LongList;
import com.almworks.integers.LongSet;
import com.almworks.jira.structure.api.attribute.*;
import com.almworks.jira.structure.api.forest.ForestSpec;
import com.almworks.jira.structure.api.pull.DataVersion;
import com.almworks.jira.structure.api.pull.VersionedDataSource;
import com.atlassian.annotations.PublicApi;
import org.jetbrains.annotations.NotNull;

import java.util.Collection;
import java.util.concurrent.*;
import java.util.function.Consumer;

/**
 * <p>{@code AttributeSubscription} represents an interest of the client code in a particular set of attribute values, allowing
 * for background loading. It acts as a buffer, collecting the loaded values and serving it to the client as a {@link VersionedDataSource}.</p>
 *
 * <h3>Defining the value set</h3>
 *
 * <p>To indicate interest in a particular set of values, you need to set the forest spec, row set and attribute set via
 * {@link #changeSubscription} method. You can later update the subscription by changing either of these parameters. The buffer
 * will be cleared and adjusted accordingly. Any background loading that is no longer required will be automatically canceled.</p>
 *
 * <p>Note that if you have interest in multiple sets (for example, in different forests, or different attributes for different rows),
 * you need to create different subscriptions, rather than constantly adjusting a single subscription.</p>
 *
 * <h3>Access and permissions</h3>
 *
 * <p>The subscription uses the public methods of {@link StructureAttributeService} to load values. All values will be loaded
 * according to the user's permissions and attribute sensitivity settings.</p>
 *
 * <p>There's no way to create a subscription that overloads security.</p>
 *
 * <h3>Loading values</h3>
 *
 * <p>Note that the values will only be loaded when you call {@link #loadValues()}. When to call this method is up to you, and you
 * can call it any time. There's no reason to throttle the calls to {@code loadValues()}, it will happen on its own inside the
 * subscription.</p>
 *
 * <p>If you never call {@link #loadValues()}, you will never receive any data!</p>
 *
 * @see AttributeSubscriptionService
 * @see StructureAttributeService
 */
@PublicApi
public interface AttributeSubscription extends VersionedDataSource<AttributeSubscriptionUpdate> {
  /**
   * Returns the ID of the subscription.
   *
   * @return subscription ID
   * @see AttributeSubscriptionService#getSubscription(Long)
   */
  long getSubscriptionId();

  /**
   * Updates the current subscription. The parts of the internal buffer that are no longer of interest will be cleared.
   * If there is any background process loading values that are no longer needed, it will be canceled.
   *
   * @param changer the code that will update the subscription
   */
  void changeSubscription(@NotNull Consumer<AttributeSubscriptionPatch> changer);

  /**
   * Returns the forest spec currently set in this subscription. When the subscription is just created, returns
   * an unspecified default forest spec.
   *
   * @return forest spec for subscribed values
   */
  @NotNull
  ForestSpec getForest();

  /**
   * Returns the attributes that are currently subscribed for.
   *
   * @return attribute specs
   */
  @NotNull
  Collection<AttributeSpec<?>> getAttributes();

  /**
   * Returns the rows that are currently subscribed for.
   *
   * @return list of row ids
   */
  @NotNull
  LongList getRows();

  /**
   * <p>A quick, non-blocking retrieval of the updated values. Calling this method is guaranteed to return <em>quickly</em>,
   * but there's no guarantee that all values will be up to date. Calling this method will not be blocked by attribute loading
   * or generating the corresponding forest.</p>
   *
   * <p>Retrieving values from {@code AttributeSubscription} follows the Pull Architecture paradigm (see {@link VersionedDataSource}).
   * The subscription acts as a buffer to contain the values loaded in the background. Once a value is loaded that is different from
   * the one that was loaded for that attribute and row before, the version of this data source is promoted and the value is stored
   * in a buffer. Requesting the values based on the previous version of the value update allows you to retrieve the values that
   * were updated since the last call.</p>
   *
   * <p>When calling this method for the first time, use {@link DataVersion#ZERO} as the version.</p>
   *
   * @param fromVersion previous version of the data, received with a previous update
   * @return an update based on the requested version
   */
  @NotNull
  @Override
  AttributeSubscriptionUpdate getUpdate(@NotNull DataVersion fromVersion);

  /**
   * <p>Requests that the subscription performs attribute loading.</p>
   *
   * <p>This will cause the subscription to start a background attribute loading process, based on the currently selected
   * forest, rows and attributes. The method will return immediately, providing the caller with a {@link Future}. This
   * {@code Future} may be waited upon to continue getting values via {@link #getUpdate} once the loading is done.</p>
   *
   * <p>The implementation may limit how many processes can be started by a single subscription. If a call to {@code loadValues()}
   * does not start a new background process, it will return the {@code future} from one of the already running processes.</p>
   *
   * <p>The returned future will be completed either with an updated version of the value update stream, or with an exception
   * if there was a problem.</p>
   *
   * <p>The future may be canceled, which will cause the loading to cancel. However, the subscription will still be usable and
   * any consequent calls to {@code loadValues()} will work.</p>
   *
   * @return future version when everything is loaded and the buffer is updated
   */
  @NotNull
  CompletableFuture<DataVersion> loadValues();

  /**
   * <p>This method can be used to check if the attribute loading is currently taking place for this subscription.</p>
   *
   * @return true if there's an active process of loading attributes
   */
  boolean isLoadingValues();

  /**
   * <p>A convenience method that calls {@link #loadValues()}, waits for the specified amount of time for the loading to complete,
   * and then provides the loaded value.</p>
   *
   * <p>Calling this method is similar to calling {@link StructureAttributeService#getAttributeValues} with a timeout.</p>
   *
   * @param timeout timeout value
   * @param unit timeout value units
   * @return loaded values
   * @throws InterruptedException if this thread was interrupted
   * @throws TimeoutException if the values haven't finished loading in the specified time
   */
  @NotNull
  RowValues getFullUpdate(long timeout, TimeUnit unit) throws InterruptedException, TimeoutException;

  /**
   * <p>A convenience method that loads all the values, waiting for them for an unspecified amount of time.</p>
   *
   * <p><strong>Use with care!</strong> When writing code that backs a user interface, always use
   * {@link #getFullUpdate(long, TimeUnit)} with a short timeout and come back for updates later using {@link #getUpdate(DataVersion)}.</p>
   *
   * @return loaded values
   */
  @NotNull
  RowValues getFullUpdate();

  /**
   * <p>Returns the last known list of rows that are not accessible by the owner of the subscription.</p>
   *
   * <p>The rows may relate to a previous window (rows and forest), so it's not correct to make an assumption that all other
   * rows are accessible. However, if there's data for a row, the row must have been accessible when the value was loaded.</p>
   *
   * @return a list of inaccessible rows
   */
  @NotNull
  LongSet getInaccessibleRows();

  /**
   * <p>Retrieves accumulated information about attribute loader errors. Provides only new errors that happened after the
   * last drain. Depending on the implementation strategy, may only return one error per attribute during the lifetime of
   * the subscription.</p>
   *
   * <p>The accumulated list of errors is cleared when this method is called.</p>
   *
   * @return the list of errors
   */
  @NotNull
  Collection<AttributeErrorInfo> drainAttributeErrors();
}
