package com.almworks.jira.structure.api.util;

import com.atlassian.crowd.embedded.api.Group;
import com.atlassian.jira.bc.EntityNotFoundException;
import com.atlassian.jira.bc.project.component.ProjectComponent;
import com.atlassian.jira.issue.Issue;
import com.atlassian.jira.issue.IssueConstant;
import com.atlassian.jira.issue.customfields.CustomFieldType;
import com.atlassian.jira.issue.customfields.option.Option;
import com.atlassian.jira.issue.fields.CustomField;
import com.atlassian.jira.issue.fields.Field;
import com.atlassian.jira.issue.issuetype.IssueType;
import com.atlassian.jira.issue.label.Label;
import com.atlassian.jira.issue.link.IssueLink;
import com.atlassian.jira.issue.link.IssueLinkType;
import com.atlassian.jira.issue.priority.Priority;
import com.atlassian.jira.issue.resolution.Resolution;
import com.atlassian.jira.issue.security.IssueSecurityLevel;
import com.atlassian.jira.issue.status.Status;
import com.atlassian.jira.issue.worklog.WorkRatio;
import com.atlassian.jira.permission.ProjectPermissions;
import com.atlassian.jira.project.Project;
import com.atlassian.jira.project.ProjectConstant;
import com.atlassian.jira.project.version.Version;
import com.atlassian.jira.security.PermissionManager;
import com.atlassian.jira.security.plugin.ProjectPermissionKey;
import com.atlassian.jira.security.roles.ProjectRole;
import com.atlassian.jira.sharing.SharedEntity;
import com.atlassian.jira.user.ApplicationUser;
import com.atlassian.jira.util.JiraKeyUtils;
import com.atlassian.jira.workflow.JiraWorkflow;
import com.atlassian.plugin.ModuleDescriptor;
import org.ofbiz.core.entity.GenericValue;

import java.security.Principal;
import java.sql.Timestamp;
import java.util.*;

import static com.almworks.jira.structure.api.item.CoreIdentities.canonicalVersionName;

/**
 * <p><code>JiraFunc</code> contains Jira-related functions.</p>
 *
 * @author Igor Sereda
 */
public class JiraFunc {
  public static final La<Project, Long> PROJECT_ID = new La<Project, Long>(Long.class) {
    @Override
    public Long la(Project argument) {
      return argument == null ? null : argument.getId();
    }
  };

  public static final La<Project, String> PROJECT_NAME = new La<Project, String>() {
    public String la(Project argument) {
      return argument == null ? null : argument.getName();
    }
  };

  public static final La<Project, String> PROJECT_KEY = new La<Project, String>(String.class) {
    public String la(Project argument) {
      return argument == null ? null : argument.getKey();
    }
  };

  public static final La<Project, String> PROJECT_DESCRIPTION = new La<Project, String>(String.class) {
    public String la(Project argument) {
      return argument == null ? null : argument.getDescription();
    }
  };

  public static final La<Issue, Long> ISSUE_ID = new La<Issue, Long>() {
    @Override
    public Long la(Issue argument) {
      return argument == null ? null : argument.getId();
    }
  };

  public static final La<String, Project> KEY_PROJECT = new La<String, Project>() {
    public Project la(String key) {
      if (key == null) return null;
      return JiraComponents.getProjectManager().getProjectObjByKey(key);
    }
  };

  public static final La<Long, Project> ID_PROJECT = new La<Long, Project>() {
    public Project la(Long id) {
      if (id == null) return null;
      return JiraComponents.getProjectManager().getProjectObj(id);
    }
  };

  public static final La<String, IssueType> ID_ISSUETYPE = new La<String, IssueType>() {
    @Override
    public IssueType la(String id) {
      return id == null ? null : JiraComponents.getConstantsManager().getIssueType(id);
    }
  };

  public static final La<String, Status> ID_STATUS = new La<String, Status>() {
    @Override
    public Status la(String id) {
      return id == null ? null : JiraComponents.getConstantsManager().getStatus(id);
    }
  };

  public static final La<String, Priority> ID_PRIORITY = new La<String, Priority>() {
    @Override
    public Priority la(String id) {
      return id == null ? null : JiraComponents.getConstantsManager().getPriorityObject(id);
    }
  };

  public static final La<String, Resolution> ID_RESOLUTION = new La<String, Resolution>() {
    @Override
    public Resolution la(String id) {
      return id == null ? null : JiraComponents.getConstantsManager().getResolution(id);
    }
  };

  public static final La<Long, ProjectComponent> ID_COMPONENT = new La<Long, ProjectComponent>() {
    @Override
    public ProjectComponent la(Long id) {
      try {
        return id == null ? null : JiraComponents.getProjectComponentManager().find(id);
      } catch (EntityNotFoundException e) {
        return null;
      }
    }
  };

  public static final La<Long, Version> ID_VERSION = new La<Long, Version>() {
    @Override
    public Version la(Long id) {
      return id == null ? null : JiraComponents.getVersionManager().getVersion(id);
    }
  };

  public static final La<ProjectRole, Long> PROJECTROLE_ID = new La<ProjectRole, Long>() {
    public Long la(ProjectRole argument) {
      return argument == null ? null : argument.getId();
    }
  };

  public static final La<Principal, String> PRINCIPAL_NAME = new La<Principal, String>(String.class) {
    public String la(Principal argument) {
      return argument == null ? null : argument.getName();
    }
  };

  public static final La<Group, String> GROUP_NAME = new La<Group, String>(String.class) {
    public String la(Group argument) {
      return argument == null ? null : argument.getName();
    }
  };

  public static final La<ModuleDescriptor, String> MODULE_COMPLETE_KEY = new La<ModuleDescriptor, String>() {
    public String la(ModuleDescriptor argument) {
      return argument == null ? null : argument.getCompleteKey();
    }
  };

  public static final La<IssueLinkType, Long> LINKTYPE_ID = new La<IssueLinkType, Long>() {
    public Long la(IssueLinkType argument) {
      return argument == null ? null : argument.getId();
    }
  };

  public static final La<SharedEntity, Long> SHAREDENTITY_ID = new La<SharedEntity, Long>() {
    public Long la(SharedEntity argument) {
      return argument == null ? null : argument.getId();
    }
  };

  public static final La<CustomField, CustomFieldType> CUSTOMFIELD_TYPE = new La<CustomField, CustomFieldType>() {
    public CustomFieldType la(CustomField argument) {
      return argument == null ? null : argument.getCustomFieldType();
    }
  };

  public static final La<CustomFieldType, String> CUSTOMFIELDTYPE_KEY = new La<CustomFieldType, String>() {
    public String la(CustomFieldType argument) {
      return argument == null ? null : argument.getKey();
    }
  };

  public static final La<CustomField, String> CUSTOMFIELD_ID = new La<CustomField, String>() {
    public String la(CustomField argument) {
      return argument == null ? null : argument.getId();
    }
  };

  public static final La<CustomField, Long> CUSTOMFIELD_LONG_ID = new La<CustomField, Long>() {
    public Long la(CustomField argument) {
      return argument == null ? null : argument.getIdAsLong();
    }
  };

  public static final La<Field, String> FIELD_NAME = new La<Field, String>(String.class) {
    @Override
    public String la(Field argument) {
      return argument == null ? null : argument.getName();
    }
  };

  public static final La<IssueType, Boolean> ISSUETYPE_SUBTASK = new La<IssueType, Boolean>() {
    public Boolean la(IssueType argument) {
      return argument != null && argument.isSubTask();
    }
  };

  public static final La<IssueConstant, String> ISSUECONSTANT_ID = new La<IssueConstant, String>(String.class) {
    public String la(IssueConstant argument) {
      return argument == null ? null : argument.getId();
    }
  };

  public static final La<IssueConstant, String> ISSUECONSTANT_NAME = new La<IssueConstant, String>(String.class) {
    @Override
    public String la(IssueConstant argument) {
      return argument == null ? null : argument.getName();
    }
  };

  public static final La<IssueConstant, String> ISSUECONSTANT_NAME_LOCAL = new La<IssueConstant, String>(String.class) {
    public String la(IssueConstant argument) {
      return argument == null ? null : argument.getNameTranslation();
    }
  };

  public static final La<IssueConstant, Long> ISSUECONSTANT_SEQUENCE = new La<IssueConstant, Long>() {
    @Override
    public Long la(IssueConstant argument) {
      return argument == null ? null : argument.getSequence();
    }
  };

  public static final La<IssueConstant, Long> NEGATED_ISSUECONSTANT_SEQUENCE = new La<IssueConstant, Long>() {
    @Override
    public Long la(IssueConstant argument) {
      return argument == null ? null : argument.getSequence() == null ? null : -argument.getSequence();
    }
  };

  public static final La<JiraWorkflow, String> WORKFLOW_NAME = new La<JiraWorkflow, String>() {
    public String la(JiraWorkflow argument) {
      return argument == null ? null : argument.getName();
    }
  };

  public static final La<ApplicationUser, String> APPLICATION_USER_NAME = new La<ApplicationUser, String>() {
    public String la(ApplicationUser argument) {
      return argument == null ? null : argument.getDisplayName();
    }
  };

  public static final La<ApplicationUser, String> USER_NAME = new La<ApplicationUser, String>() {
    public String la(ApplicationUser argument) {
      return argument == null ? null : argument.getDisplayName();
    }
  };

  public static final La<ApplicationUser, String> USER_KEY = new La<ApplicationUser, String>() {
    public String la(ApplicationUser argument) {
      return argument == null ? null : argument.getKey();
    }
  };

  public static final La<Version, Boolean> VERSION_ACTIVE = new La<Version, Boolean>(Boolean.class) {
    @Override
    public Boolean la(Version argument) {
      return argument == null ? null : !argument.isArchived();
    }
  };

  public static final La<Version, Long> VERSION_SEQUENCE = new La<Version, Long>(Long.class) {
    @Override
    public Long la(Version argument) {
      return argument == null ? null : argument.getSequence();
    }
  };

  public static final La<Version, Project> VERSION_PROJECT = new La<Version, Project>(Project.class) {
    @Override
    public Project la(Version argument) {
      return argument == null ? null : argument.getProject();
    }
  };

  public static final La<ProjectConstant, Long> PROJECTCONSTANT_ID = new La<ProjectConstant, Long>(Long.class) {
    @Override
    public Long la(ProjectConstant argument) {
      return argument == null ? null : argument.getId();
    }
  };

  public static final La<ProjectConstant, String> PROJECTCONSTANT_NAME = new La<ProjectConstant, String>(String.class) {
    @Override
    public String la(ProjectConstant argument) {
      return argument == null ? null : argument.getName();
    }
  };

  public static final La<ProjectConstant, String> PROJECTCONSTANT_DESCRIPTION = new La<ProjectConstant, String>(String.class) {
    @Override
    public String la(ProjectConstant argument) {
      return argument == null ? null : argument.getDescription();
    }
  };

  public static final La<ProjectConstant, Long> PROJECTCONSTANT_PROJECTID = new La<ProjectConstant, Long>(Long.class) {
    @Override
    public Long la(ProjectConstant argument) {
      return argument == null ? null : argument.getProjectId();
    }
  };

  public static final La<GenericValue, Long> GENERICVALUE_LONG_ID = new La<GenericValue, Long>() {
    public Long la(GenericValue argument) {
      try {
        return argument == null ? null : argument.getLong("id");
      } catch (Exception e) {
        return null;
      }
    }
  };

  public static final La<GenericValue, String> GENERICVALUE_STRING_ID = new La<GenericValue, String>() {
    public String la(GenericValue argument) {
      try {
        return argument == null ? null : argument.getString("id");
      } catch (Exception e) {
        return null;
      }
    }
  };

  public static final La<Issue, Project> ISSUE_PROJECT = new La<Issue, Project>() {
    public Project la(Issue argument) {
      return argument == null ? null : argument.getProjectObject();
    }
  };

  public static final La<Issue, Long> ISSUE_PROJECTID = new La<Issue, Long>(Long.class) {
    @Override
    public Long la(Issue argument) {
      return argument == null ? null : argument.getProjectId();
    }
  };

  public static final La<Issue, IssueType> ISSUE_ISSUETYPE = new La<Issue, IssueType>() {
    public IssueType la(Issue argument) {
      return argument == null ? null : argument.getIssueType();
    }
  };

  public static final La<IssueType, String> ISSUETYPE_ID = new La<IssueType, String>() {
    public String la(IssueType argument) {
      return argument == null ? null : argument.getId();
    }
  };

  public static final La<IssueType, String> ISSUETYPE_NAME = new La<IssueType, String>() {
    public String la(IssueType argument) {
      return argument == null ? null : argument.getNameTranslation();
    }
  };

  public static final La<Issue, Priority> ISSUE_PRIORITY = new La<Issue, Priority>() {
    public Priority la(Issue argument) {
      return argument == null ? null : argument.getPriority();
    }
  };

  public static final La<Issue, Status> ISSUE_STATUS = new La<Issue, Status>() {
    @Override
    public Status la(Issue argument) {
      return argument == null ? null : argument.getStatus();
    }
  };

  public static final La<Issue, Resolution> ISSUE_RESOLUTION = new La<Issue, Resolution>() {
    @Override
    public Resolution la(Issue argument) {
      return argument == null ? null : argument.getResolution();
    }
  };

  public static final La<Issue, String> ISSUE_ISSUETYPEID = ISSUE_ISSUETYPE.supply(ISSUETYPE_ID);

  public static final La<Issue, String> ISSUE_PRIORITYID = ISSUE_PRIORITY.supply(ISSUECONSTANT_ID);

  public static final La<Issue, String> ISSUE_RESOLUTIONID = ISSUE_RESOLUTION.supply(ISSUECONSTANT_ID);

  public static final La<Issue, String> ISSUE_STATUSID = ISSUE_STATUS.supply(ISSUECONSTANT_ID);

  public static final La<Issue, String> ISSUE_SUMMARY = new La<Issue, String>(String.class) {
    @Override
    public String la(Issue argument) {
      return argument == null ? null : argument.getSummary();
    }
  };

  public static final La<Issue, String> ISSUE_DESCRIPTION = new La<Issue, String>(String.class) {
    @Override
    public String la(Issue argument) {
      return argument == null ? null : argument.getDescription();
    }
  };

  public static final La<Issue, String> ISSUE_ENVIRONMENT = new La<Issue, String>(String.class) {
    @Override
    public String la(Issue argument) {
      return argument == null ? null : argument.getEnvironment();
    }
  };

  public static final La<Issue, String> ISSUE_KEY = new La<Issue, String>(String.class) {
    @Override
    public String la(Issue argument) {
      return argument == null ? null : argument.getKey();
    }
  };

  public static final La<Issue, Long> ISSUE_ORIGINAL_ESTIMATE = new La<Issue, Long>(Long.class) {
    @Override
    public Long la(Issue issue) {
      return issue == null ? null : issue.getOriginalEstimate();
    }
  };

  public static final La<Issue, Long> ISSUE_REMAINING_ESTIMATE = new La<Issue, Long>(Long.class) {
    @Override
    public Long la(Issue issue) {
      return issue == null ? null : issue.getEstimate();
    }
  };

  public static final La<Issue, Long> ISSUE_TIME_SPENT = new La<Issue, Long>(Long.class) {
    @Override
    public Long la(Issue issue) {
      return issue == null ? null : issue.getTimeSpent();
    }
  };

  public static final La<Issue, Long> ISSUE_TOTAL_TIME = new La<Issue, Long>(Long.class) {
    @Override
    public Long la(Issue issue) {
      Long spent = issue == null ? null : issue.getTimeSpent();
      Long remaining =  issue == null ? null : issue.getEstimate();
      Long total = NumericFunctions.longAddOrNull(spent, remaining);
      return total != null && total > 0 ? total : null;
    }
  };

  public static final La<Issue, Long> ISSUE_WORK_RATIO = new La<Issue, Long>() {
    public Long la(Issue argument) {
      return argument == null ? null : WorkRatio.getWorkRatio(argument);
    }
  };

  public static final La<Issue, Long> ISSUE_VOTES = new La<Issue, Long>(Long.class) {
    @Override
    public Long la(Issue argument) {
      return argument == null ? null : argument.getVotes();
    }
  };

  public static final La<Issue, Long> ISSUE_WATCHES = new La<Issue, Long>(Long.class) {
    @Override
    public Long la(Issue argument) {
      return argument == null ? null : argument.getWatches();
    }
  };

  public static final La<String, ApplicationUser> USERKEY_APPLICATIONUSER = new La<String, ApplicationUser>(ApplicationUser.class) {
    @Override
    public ApplicationUser la(String argument) {
      return JiraUsers.byKey(argument);
    }
  };

  public static final La<Issue, String> ISSUE_ASSIGNEEID = new La<Issue, String>(String.class) {
    @Override
    public String la(Issue argument) {
      return argument == null ? null : argument.getAssigneeId();
    }
  };

  public static final La<Issue, ApplicationUser> ISSUE_ASSIGNEE = USERKEY_APPLICATIONUSER.apply(ISSUE_ASSIGNEEID);

  public static final La<Issue, String> ISSUE_REPORTERID = new La<Issue, String>(String.class) {
    @Override
    public String la(Issue argument) {
      return argument == null ? null : argument.getReporterId();
    }
  };

  public static final La<Issue, ApplicationUser> ISSUE_REPORTER = USERKEY_APPLICATIONUSER.apply(ISSUE_REPORTERID);

  public static final La<Issue, String> ISSUE_CREATORID = new La<Issue, String>(String.class) {
    public String la(Issue argument) {
      return argument == null ? null : argument.getCreatorId();
    }
  };

  public static final La<Issue, ApplicationUser> ISSUE_CREATOR = USERKEY_APPLICATIONUSER.apply(ISSUE_CREATORID);

  public static final La<Issue, Timestamp> ISSUE_CREATED = new La<Issue, Timestamp>(Timestamp.class) {
    @Override
    public Timestamp la(Issue argument) {
      return argument == null ? null : argument.getCreated();
    }
  };

  public static final La<Issue, Timestamp> ISSUE_UPDATED = new La<Issue, Timestamp>(Timestamp.class) {
    @Override
    public Timestamp la(Issue argument) {
      return argument == null ? null : argument.getUpdated();
    }
  };

  public static final La<Issue, Timestamp> ISSUE_DUE_DATE = new La<Issue, Timestamp>(Timestamp.class) {
    @Override
    public Timestamp la(Issue argument) {
      return argument == null ? null : argument.getDueDate();
    }
  };

  public static final La<Issue, Timestamp> ISSUE_RESOLUTION_DATE = new La<Issue, Timestamp>(Timestamp.class) {
    @Override
    public Timestamp la(Issue argument) {
      return argument == null ? null : argument.getResolutionDate();
    }
  };

  public static final La<Issue, Collection<Version>> ISSUE_AFFECTED_VERSIONS = new La<Issue, Collection<Version>>() {
    @Override
    public Collection<Version> la(Issue argument) {
      return argument == null ? null : argument.getAffectedVersions();
    }
  };

  public static final La<Issue, Collection<Version>> ISSUE_FIX_VERSIONS = new La<Issue, Collection<Version>>() {
    @Override
    public Collection<Version> la(Issue argument) {
      return argument == null ? null : argument.getFixVersions();
    }
  };

  public static final La<Issue, Collection<ProjectComponent>> ISSUE_COMPONENTS = new La<Issue, Collection<ProjectComponent>>() {
    @Override
    public Collection<ProjectComponent> la(Issue argument) {
      return argument == null ? null : argument.getComponents();
    }
  };

  public static final La<Issue, Set<Label>> ISSUE_LABELS = new La<Issue, Set<Label>>() {
    @Override
    public Set<Label> la(Issue argument) {
      return argument == null ? null : argument.getLabels();
    }
  };

  public static final La<IssueLink, Long> LINK_ID = new La<IssueLink, Long>() {
    @Override
    public Long la(IssueLink argument) {
      return argument == null ? null : argument.getId();
    }
  };

  public static final La<IssueLink, Long> LINK_SEQUENCE = new La<IssueLink, Long>() {
    @Override
    public Long la(IssueLink argument) {
      return argument == null ? null : argument.getSequence();
    }
  };

  public static final La<Label, String> LABEL_LABEL = new La<Label, String>(String.class) {
    @Override
    public String la(Label argument) {
      return argument == null ? null : argument.getLabel();
    }
  };

  public static final La<IssueSecurityLevel, Long> SECURITY_LEVEL_ID = new La<IssueSecurityLevel, Long>(Long.class) {
    @Override
    public Long la(IssueSecurityLevel argument) {
      return argument == null ? null : argument.getId();
    }
  };

  public static final La<IssueSecurityLevel, String> SECURITY_LEVEL_NAME = new La<IssueSecurityLevel, String>(String.class) {
    @Override
    public String la(IssueSecurityLevel argument) {
      return argument == null ? null : argument.getName();
    }
  };

  public static La<Project, Boolean> browseableBy(ApplicationUser user) {
    return accessibleBy(user, ProjectPermissions.BROWSE_PROJECTS);
  }

  public static La<Project, Boolean> accessibleBy(final ApplicationUser user, final ProjectPermissionKey permission) {
    final PermissionManager permissionManager = JiraComponents.getPermissionManager();
    if (permissionManager == null) {
      assert false : "no permissions";
      return La.constant(false);
    }
    return new La<Project, Boolean>(Boolean.class) {
      @Override
      public Boolean la(Project argument) {
        return argument != null && permissionManager.hasPermission(permission, argument, user);
      }
    };
  }

  public static La<Project, Boolean> hasCustomField(final CustomField field) {
    if (field == null) return La.constant(false);
    List<Project> projects = field.isAllProjects() ? null : field.getAssociatedProjectObjects();
    final Set<Long> pids = projects == null ? null : PROJECT_ID.hashSet(projects);
    return new La<Project, Boolean>() {
      public Boolean la(Project argument) {
        return argument != null && (pids == null || pids.contains(argument.getId()));
      }
    };
  }

  public static final La<String, String> VALID_ISSUE_KEY = new La<String, String>() {
    @Override
    public String la(String issueKey) {
      return issueKey != null && JiraKeyUtils.validIssueKey(issueKey) ? issueKey : null;
    }
  };

  public static final La<Option, String> OPTION_NAME = new La<Option, String>() {
    @Override
    public String la(Option argument) {
      if (argument != null) {
        Option parent = argument.getParentOption();
        if (parent != null) {
          return parent.getValue() + " - " + argument.getValue();
        }
        return argument.getValue();
      }
      return null;
    }
  };

  public static final La<Option, Long> OPTION_SEQUENCE = new La<Option, Long>() {
    @Override
    public Long la(Option option) {
      return option == null ? null : option.getSequence();
    }
  };

  public static final La<Option, Long> OPTION_ID = new La<Option, Long>(Long.class) {
    @Override
    public Long la(Option argument) {
      return argument == null ? null : argument.getOptionId();
    }
  };

  public static final La<Long, Option> ID_OPTION = new La<Long, Option>() {
    @Override
    public Option la(Long id) {
      return id == null ? null : JiraComponents.getOptionsManager().findByOptionId(id);
    }
  };

  public static final La<Version, String> CANONICAL_VERSION_NAME = new La<Version, String>() {
    @Override
    public String la(Version v) {
      return canonicalVersionName(PROJECTCONSTANT_NAME.la(v));
    }
  };
}