package com.almworks.jira.structure.api.job;

import com.atlassian.jira.user.ApplicationUser;

/**
 * <p><code>StructureJob</code> is a single one-time or repetitive background task,
 * run by the {@link StructureJobManager}.</p>
 *
 * <p>The interface defines the main {@link #job} method that does the work and a few other
 * methods to provide the lifecycle information for the job manager.</p>
 *
 * <p>Typically you should extend {@link AbstractStructureJob} to have housekeeping done for you.</p>
 *
 * <p>All methods of this interface must be implemented in a thread-safe way, because
 * typically maintenance and running the job happen in different threads. However, the
 * implementation may be certain the <code>job()</code> method (or {@link AbstractStructureJob#doJob()} method)
 * will never be called concurrently in different threads.</p>
 *
 * @see AbstractStructureJob
 * @see StructureJobManager
 * @author Igor Sereda
 */
public interface StructureJob {
  /**
   * <p>Runs the job in background. Before this method is called, all JIRA thread maintenance is done
   * and the current user is set up to be what {@link #getUser()} returns.</p>
   *
   * @throws Exception any exception thrown by the job will be logged but won't stop job manager
   */
  void job() throws Exception;

  /**
   * <p>Reset the state of the job so that it becomes {@link State#PENDING}. Should not be called
   * when the job is running.</p>
   */
  void reset();

  /**
   * @return the current state of the job
   * @see State
   */
  State getState();

  /**
   * @return the JIRA user account, under which the job must run, null means anonymous
   */
  ApplicationUser getUser();

  /**
   * Simple enumeration of the possible job states
   */
  enum State {
    /**
     * The job has been scheduled and in the queue
     */
    PENDING,

    /**
     * The job is running now
     */
    RUNNING,

    /**
     * The job has finished and will no longer execute
     */
    FINISHED
  }
}
