package com.almworks.jira.structure.api.generator.util;

import com.almworks.integers.*;
import com.almworks.jira.structure.api.generator.ItemChangeFilter;
import com.almworks.jira.structure.api.generator.StructureGenerator;
import com.almworks.jira.structure.api.item.ItemIdentity;
import com.almworks.jira.structure.api.row.StructureRow;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.*;

import static com.almworks.jira.structure.api.item.CoreIdentities.isIssue;

public class BasicItemChangeFilter implements RecordingItemChangeFilter {
  private final LongSet myReadonlyIssues;
  @Nullable
  private final WritableLongSet myWritableIssues;
  private final Set<ItemIdentity> myNonIssues;

  private BasicItemChangeFilter(LongSet readonlyIssues, @Nullable WritableLongSet writableIssues,
    Set<ItemIdentity> nonIssues)
  {
    assert (writableIssues == null) ^ (writableIssues == readonlyIssues);
    myReadonlyIssues = readonlyIssues;
    myWritableIssues = writableIssues;
    myNonIssues = nonIssues;
  }

  protected BasicItemChangeFilter(WritableLongSet writableIssues, Set<ItemIdentity> writableNonIssues) {
    this(writableIssues, writableIssues, writableNonIssues);
  }
  
  public static ItemChangeFilter forIssues(@NotNull LongSet issues) {
    return new BasicItemChangeFilter(issues, null, Collections.<ItemIdentity>emptySet());
  }

  public static ItemChangeFilter forItems(@NotNull LongSet issues, Set<ItemIdentity> nonIssues) {
    return new BasicItemChangeFilter(issues, null, nonIssues);
  }
  
  public static RecordingItemChangeFilter createRecording() {
    return new BasicItemChangeFilter(new LongOpenHashSet(), new HashSet<ItemIdentity>());
  }
  
  public static RecordingItemChangeFilter createRecording(ItemIdentity... items) {
    RecordingItemChangeFilter recording = createRecording();
    for (ItemIdentity item : items) {
      recording.recordItem(item);
    }
    return recording;
  }

  @Override
  public final void recordItem(StructureRow row) {
    ItemIdentity itemId = row.getItemId();
    recordItem(itemId);
  }

  public final void recordItem(ItemIdentity itemId) {
    if (myWritableIssues != null) {
      if (isIssue(itemId)) {
        myWritableIssues.add(itemId.getLongId());
      } else {
        myNonIssues.add(itemId);
      }
    } else {
      throw new UnsupportedOperationException();
    }
  }

  @Override
  public boolean accept(@NotNull Set<ItemIdentity> itemIds, @NotNull StructureGenerator.ItemChangeFilterContext context) {
    for (ItemIdentity itemId : itemIds) {
      if (isIssue(itemId)) {
        if (myReadonlyIssues.contains(itemId.getLongId())) {
          return true;
        }
      } else {
        if (myNonIssues.contains(itemId)) {
          return true;
        }
      }
    }
    return false;
  }
}
