package com.almworks.jira.structure.api.effector.instance;

import com.almworks.jira.structure.api.error.StructureException;
import com.atlassian.annotations.PublicApi;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Map;

/**
 * Effector instance manager is responsible for CRUD operations on persistent
 * {@link EffectorInstance effector instances}.
 */
@PublicApi
public interface EffectorInstanceManager {
  /**
   * Creates a new effector instance and returns its ID.
   *
   * @param moduleKey the module key of the {@link com.almworks.jira.structure.api.effector.Effector}
   * @param parameters effector-specific parameters
   * @param owningStructure the ID of the structure this instance will be
   * installed into, if applicable. It's always safe to pass {@code null} here,
   * Structure will set this parameter when needed.
   * @return the ID of the created instance
   * @throws StructureException if an effector instance cannot be created
   */
  long createEffectorInstance(@NotNull String moduleKey, @Nullable Map<String, Object> parameters,
    @Nullable Long owningStructure) throws StructureException;

  /**
   * Returns an effector instance by its ID.
   *
   * @param id instance ID
   * @return the corresponding effector instance
   * @throws StructureException if an effector instance with this ID does not
   * exist or cannot be returned
   */
  @NotNull
  EffectorInstance getEffectorInstance(long id) throws StructureException;

  /**
   * Updates an effector instance with the given ID.
   *
   * @param id instance ID
   * @param moduleKey the module key of the {@link com.almworks.jira.structure.api.effector.Effector}
   * @param parameters effector-specific parameters
   * @param owningStructure the ID of the structure this instance will be
   * installed into, if applicable.
   * @throws StructureException if an effector instance with this ID does not
   * exist or cannot be updated
   */
  void updateEffectorInstance(long id, @NotNull String moduleKey, @Nullable Map<String, Object> parameters,
    @Nullable Long owningStructure) throws StructureException;

  /**
   * Deletes an effector instance with the given ID.
   *
   * @param id instance ID
   * @throws StructureException if an effector instance cannot be deleted
   */
  void deleteEffectorInstance(long id) throws StructureException;
}
