package com.almworks.jira.structure.api.attribute.loader.composition;

import com.almworks.integers.LongSet;
import com.almworks.jira.structure.api.attribute.AttributeValue;
import com.almworks.jira.structure.api.attribute.loader.*;
import com.almworks.jira.structure.api.attribute.loader.delegate.DelegatingAttributeLoader;
import com.almworks.jira.structure.api.forest.item.ItemForest;
import com.almworks.jira.structure.api.item.ItemIdentity;
import com.almworks.jira.structure.api.row.ItemAccessMode;
import com.almworks.jira.structure.api.row.StructureRow;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.*;
import java.util.function.BiFunction;
import java.util.function.Function;

public abstract class AttributeLoaderAdapter<T, L extends AttributeLoader<T>> extends DelegatingAttributeLoader<T, L> {
  private AttributeLoaderAdapter(@NotNull L loader) {
    super(loader);
  }

  static <T> Function<AttributeLoader<T>, AttributeLoader<T>> tryAdaptLoader(LoaderType targetLoaderType) {
    switch (targetLoaderType) {
    case ITEM:
      return loader -> {
        switch (LoaderType.getType(loader)) {
        case DERIVED:
          return new DerivedToItem<>((DerivedAttributeLoader<T>) loader);
        default:
          return loader;
        }
      };
    case SINGLE_ROW:
      return loader -> {
        switch (LoaderType.getType(loader)) {
        case DERIVED:
          return new DerivedToSingleRow<>((DerivedAttributeLoader<T>) loader);
        case ITEM:
          return new ItemToSingleRow<>((ItemAttributeLoader<T>) loader);
        default:
          return loader;
        }
      };
    case AGGREGATE:
      return loader -> {
        switch (LoaderType.getType(loader)) {
        case DERIVED:
          return new DerivedToAggregate<>((DerivedAttributeLoader<T>) loader);
        case ITEM:
          return new ItemToAggregate<>((ItemAttributeLoader<T>) loader);
        case SINGLE_ROW:
          return new SingleRowToAggregate<>((SingleRowAttributeLoader<T>) loader);
        default:
          return loader;
        }
      };
    case PROPAGATE:
      return loader -> {
        switch (LoaderType.getType(loader)) {
        case DERIVED:
          return new DerivedToPropagate<>((DerivedAttributeLoader<T>) loader);
        case ITEM:
          return new ItemToPropagate<>((ItemAttributeLoader<T>) loader);
        case SINGLE_ROW:
          return new SingleRowToPropagate<>((SingleRowAttributeLoader<T>) loader);
        default:
          return loader;
        }
      };
    case SCANNING:
      return loader -> {
        switch (LoaderType.getType(loader)) {
        case DERIVED:
          return new DerivedToScanning<>((DerivedAttributeLoader<T>) loader);
        case ITEM:
          return new ItemToScanning<>((ItemAttributeLoader<T>) loader);
        case SINGLE_ROW:
          return new SingleRowToScanning<>((SingleRowAttributeLoader<T>) loader);
        default:
          return loader;
        }
      };
    default:
      return loader -> loader;
    }
  }

  public static <T, C extends AttributeLoader<T>> C adapt(@NotNull AttributeLoader<T> loader, @NotNull Class<C> clazz) {
    LoaderType currentType = LoaderType.getType(loader);
    LoaderType wantedType = LoaderType.getType(clazz);
    if (currentType == wantedType) {
      return clazz.cast(loader);
    }
    Function<AttributeLoader<T>, AttributeLoader<T>> f = tryAdaptLoader(wantedType);
    AttributeLoader<T> adapted = f.apply(loader);
    if (adapted == loader) {
      throw new IllegalArgumentException("unadaptable loaders");
    }
    return clazz.cast(adapted);
  }

  L getAdaptedLoader() {
    return delegate();
  }


  private static class DerivedToAggregate<T> extends AttributeLoaderAdapter<T, DerivedAttributeLoader<T>> implements AggregateAttributeLoader<T> {
    DerivedToAggregate(DerivedAttributeLoader<T> loader) {
      super(loader);
    }

    @Override
    public AttributeValue<T> loadValue(@NotNull List<AttributeValue<T>> childrenValues, @NotNull AggregateAttributeContext context) {
      assert context instanceof DerivedAttributeContext : context;
      DerivedAttributeContext derivedContext = (DerivedAttributeContext) context;
      return delegate().loadValue(derivedContext);
    }
  }


  private static class DerivedToItem<T> extends AttributeLoaderAdapter<T, DerivedAttributeLoader<T>> implements ItemAttributeLoader<T> {
    DerivedToItem(DerivedAttributeLoader<T> loader) {
      super(loader);
    }

    @Override
    public boolean isItemTypeSupported(@NotNull String itemType) {
      return true;
    }

    @Nullable
    @Override
    public AttributeValue<T> loadValue(@NotNull ItemIdentity itemId, @NotNull ItemAttributeContext context) {
      assert context instanceof DerivedAttributeContext : context;
      DerivedAttributeContext derivedContext = (DerivedAttributeContext) context;
      return delegate().loadValue(derivedContext);
    }
  }


  private static class DerivedToPropagate<T> extends AttributeLoaderAdapter<T, DerivedAttributeLoader<T>> implements PropagateAttributeLoader<T> {
    DerivedToPropagate(DerivedAttributeLoader<T> loader) {
      super(loader);
    }

    @Nullable
    @Override
    public BiFunction<StructureRow, PropagateAttributeContext, AttributeValue<T>> loadChildren(@NotNull AttributeValue<T> parentValue,
      @NotNull PropagateAttributeContext.Parent context)
    {
      return this::loadChild;
    }

    private AttributeValue<T> loadChild(StructureRow row, PropagateAttributeContext rowContext) {
      assert rowContext instanceof DerivedAttributeContext : rowContext;
      DerivedAttributeContext derivedContext = (DerivedAttributeContext) rowContext;
      return delegate().loadValue(derivedContext);
    }
  }


  private static class DerivedToSingleRow<T> extends AttributeLoaderAdapter<T, DerivedAttributeLoader<T>> implements SingleRowAttributeLoader<T> {
    DerivedToSingleRow(DerivedAttributeLoader<T> loader) {
      super(loader);
    }

    @Nullable
    @Override
    public AttributeValue<T> loadValue(@NotNull StructureRow row, @NotNull SingleRowAttributeContext context) {
      assert context instanceof DerivedAttributeContext : context;
      DerivedAttributeContext derivedContext = (DerivedAttributeContext) context;
      return delegate().loadValue(derivedContext);
    }
  }


  private static class DerivedToScanning<T> extends AttributeLoaderAdapter<T, DerivedAttributeLoader<T>> implements ScanningAttributeLoader<T> {
    DerivedToScanning(DerivedAttributeLoader<T> loader) {
      super(loader);
    }

    @Nullable
    @Override
    public AttributeValue<T> loadValue(@NotNull AttributeValue<T> precedingValue, @NotNull ScanningAttributeContext context) {
      assert context instanceof DerivedAttributeContext : context;
      DerivedAttributeContext derivedContext = (DerivedAttributeContext) context;
      return delegate().loadValue(derivedContext);
    }
  }


  private static abstract class ItemToRow<T> extends AttributeLoaderAdapter<T, ItemAttributeLoader<T>> implements RowAttributeLoader<T> {
    ItemToRow(ItemAttributeLoader<T> loader) {
      super(loader);
    }

    @Override
    public void preload(@NotNull LongSet rowIds, @NotNull ItemForest forest, @NotNull AttributeContext context) {
      Set<ItemIdentity> idSet = new HashSet<>();
      forest.scanAllExistingRows(rowIds, false, ItemAccessMode.ITEM_NOT_NEEDED, row -> {
        ItemIdentity itemId = row.getItemId();
        if (delegate().isItemTypeSupported(itemId.getItemType())) {
          idSet.add(itemId);
        }
      });
      delegate().preload(idSet, context);
    }

    protected AttributeValue<T> loadItemValue(RowAttributeContext context) {
      assert context instanceof ItemAttributeContext : context;
      ItemAttributeContext itemContext = (ItemAttributeContext) context;
      ItemIdentity itemId = itemContext.getItemId();
      if (delegate().isItemTypeSupported(itemId.getItemType())) {
        return delegate().loadValue(itemId, itemContext);
      }
      return null;
    }
  }


  private static class ItemToAggregate<T> extends ItemToRow<T> implements AggregateAttributeLoader<T> {
    ItemToAggregate(ItemAttributeLoader<T> loader) {
      super(loader);
    }

    @Override
    public AttributeValue<T> loadValue(@NotNull List<AttributeValue<T>> childrenValues, @NotNull AggregateAttributeContext context) {
      return loadItemValue(context);
    }
  }


  private static class ItemToPropagate<T> extends ItemToRow<T> implements PropagateAttributeLoader<T> {
    ItemToPropagate(ItemAttributeLoader<T> loader) {
      super(loader);
    }

    @Nullable
    @Override
    public BiFunction<StructureRow, PropagateAttributeContext, AttributeValue<T>> loadChildren(@NotNull AttributeValue<T> parentValue,
      @NotNull PropagateAttributeContext.Parent context)
    {
      return (row, rowContext) -> loadItemValue(rowContext);
    }
  }


  private static class ItemToSingleRow<T> extends ItemToRow<T> implements SingleRowAttributeLoader<T> {
    ItemToSingleRow(ItemAttributeLoader<T> loader) {
      super(loader);
    }

    @Nullable
    @Override
    public AttributeValue<T> loadValue(@NotNull StructureRow row, @NotNull SingleRowAttributeContext context) {
      return loadItemValue(context);
    }
  }


  private static class ItemToScanning<T> extends ItemToRow<T> implements ScanningAttributeLoader<T> {
    ItemToScanning(ItemAttributeLoader<T> loader) {
      super(loader);
    }

    @Nullable
    @Override
    public AttributeValue<T> loadValue(@NotNull AttributeValue<T> precedingValue, @NotNull ScanningAttributeContext context) {
      return loadItemValue(context);
    }
  }


  private static abstract class RowToRow<T, L extends RowAttributeLoader<T>> extends AttributeLoaderAdapter<T, L> implements RowAttributeLoader<T> {
    RowToRow(@NotNull L loader) {
      super(loader);
    }

    @Override
    public void preload(@NotNull LongSet rowIds, @NotNull ItemForest forest, @NotNull AttributeContext context) {
      delegate().preload(rowIds, forest, context);
    }

    @Override
    public boolean isWholeForestDependent() {
      return delegate().isWholeForestDependent();
    }
  }


  private static class SingleRowToAggregate<T> extends RowToRow<T, SingleRowAttributeLoader<T>> implements AggregateAttributeLoader<T> {
    SingleRowToAggregate(SingleRowAttributeLoader<T> loader) {
      super(loader);
    }

    @Override
    public AttributeValue<T> loadValue(@NotNull List<AttributeValue<T>> childrenValues, @NotNull AggregateAttributeContext context) {
      assert context instanceof SingleRowAttributeContext : context;
      SingleRowAttributeContext singleRowContext = (SingleRowAttributeContext) context;
      return delegate().loadValue(singleRowContext.getRow(), singleRowContext);
    }
  }


  private static class SingleRowToPropagate<T> extends AttributeLoaderAdapter<T, SingleRowAttributeLoader<T>> implements PropagateAttributeLoader<T> {
    SingleRowToPropagate(SingleRowAttributeLoader<T> loader) {
      super(loader);
    }

    @Nullable
    @Override
    public BiFunction<StructureRow, PropagateAttributeContext, AttributeValue<T>> loadChildren(@NotNull AttributeValue<T> parentValue,
      @NotNull PropagateAttributeContext.Parent context)
    {
      return (row, rowContext) -> {
        assert rowContext instanceof SingleRowAttributeContext : rowContext;
        SingleRowAttributeContext singleRowContext = (SingleRowAttributeContext) rowContext;
        return delegate().loadValue(row, singleRowContext);
      };
    }
  }


  private static class SingleRowToScanning<T> extends RowToRow<T, SingleRowAttributeLoader<T>> implements ScanningAttributeLoader<T> {
    SingleRowToScanning(SingleRowAttributeLoader<T> loader) {
      super(loader);
    }

    @Nullable
    @Override
    public AttributeValue<T> loadValue(@NotNull AttributeValue<T> precedingValue, @NotNull ScanningAttributeContext context) {
      assert context instanceof SingleRowAttributeContext : context;
      SingleRowAttributeContext singleRowContext = (SingleRowAttributeContext) context;
      return delegate().loadValue(singleRowContext.getRow(), singleRowContext);
    }
  }
}
