package com.almworks.jira.structure.api.sync;

import com.almworks.jira.structure.api.permissions.PermissionLevel;
import com.atlassian.annotations.PublicApi;
import com.atlassian.jira.user.ApplicationUser;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.List;

/**
 * <p><code>StructureSyncManager</code> manages the whole synchronization engine.</p>
 *
 * <p>There are several synchronizers in the system, provided by &lt;structure-synchronizer&gt;
 * modules, and each synchronizer may have several instances, represented by {@link SyncInstance}.
 * Synchronizer instance is a configured synchronizer, installed onto a specific structure. Synchronizer
 * instances are addressed via unique ID.
 * </p>
 * 
 * <p>This class does not check permissions of the caller.
 * If the calling code works on behalf of a user, it must check whether
 * the user has <code>{@link PermissionLevel#ADMIN}</code> permission level on the subject structure.
 * </p>
 * 
 * @author Igor Sereda
 */
@PublicApi
public interface StructureSyncManager {
  /**
   * Retrieves an instance of installed synchronizer.
   *
   * @param instanceId the ID of the instance
   * @return synchronizer instance, or null if there's no instance with such ID
   */
  @Nullable
  SyncInstance getInstalledSynchronizer(@Nullable Long instanceId);

  /**
   * Retrieves all synchronizer instances installed for the given structure.
   *
   * @param structureId the ID of the structure
   * @return a list of installed synchronizer instances
   */
  @NotNull
  List<SyncInstance> getInstalledSynchronizersForStructure(@Nullable Long structureId);

  /**
   * Installs a new synchronizer for a structure. Synchronizer configuration is persisted -
   * it will be there after JIRA restart - but the synchronizer is not yet enabled for
   * incremental synchronization. To enable the synchronizer, use {@link #setAutosyncEnabled}
   * or {@link #resync}.
   *
   * @param synchronizer the type of the synchronizer to be installed
   * @param structureId the ID of the structure
   * @param params synchronization parameters, of the type accepted by the synchronizer
   * @param user the user account, under which the synchronization will run. {@code null} (anonymous) is not allowed
   * @return the new instance of the synchronizer
   * @throws StructureSynchronizerException if the synchronizer cannot be installed
   */
  @NotNull
  SyncInstance installSynchronizer(@NotNull StructureSynchronizer synchronizer, @NotNull Long structureId,
    @Nullable Object params, @Nullable ApplicationUser user) throws StructureSynchronizerException;

  /**
   * <p>Installs synchronizer based on the serialized data. The effect is the same as
   * with {@link #installSynchronizer}, but the parameters allow to define synchronizer
   * without resolving data into the classes required by the <code>installSynchronizer</code>
   * method.</p>
   *
   * <p>This method also allows installing a synchronizer that is currently not present
   * in the system - for example, if a plugin that provides it is disabled. The synchronizer
   * will be installed but {@link SyncInstance} will not be created until the synchronizer
   * module is available.</p>
   *
   *
   * @param moduleKey the full module key of the synchronizer module
   * @param parameters serialized parameters (as previously returned by {@link StructureSynchronizer#storeParameters(Object)}).
   *   Note that empty String is treated the same way as {@code null}.
   * @param structureId the ID of the structure
   * @param ownerKey user key of the synchronizer owner
   * @param autosyncEnabled if true, the autosync should be initially enabled for this structure
   * @throws StructureSynchronizerException if the synchronizer cannot be installed
   * @return the ID of the installed synchronizer instance
   */
  long installSynchronizerOffline(@NotNull String moduleKey, @Nullable String parameters,
    @NotNull Long structureId, @Nullable String ownerKey, boolean autosyncEnabled) throws StructureSynchronizerException;

  /**
   * Disables and uninstalls synchronizer with the specified ID. If the synchronization is running, it may finish.
   *
   * @param instanceId the ID of the synchronizer
   */
  void uninstallSynchronizer(@Nullable Long instanceId);

  /**
   * Disables and uninstalls all synchronizers for all structures.
   */
  void uninstallAllSynchronizers();

  /**
   * Convenience method that disables and uninstalls all synchronizers for the specified structure.
   * @param structureId the ID of the structure
   */
  void uninstallSynchronizersForStructure(@Nullable Long structureId);

  /**
   * Checks if incremental synchronization is enabled for the specified synchronizer
   *
   * @param instanceId the ID of the synchronizer instance
   * @return true if the synchronizer is present and has autosync enabled
   */
  boolean isAutosyncEnabled(@Nullable Long instanceId);

  /**
   * Enables or disables the incremental synchronization for the specified synchronizer.
   *
   * @param instanceId the ID of the synchronizer instance
   * @param enabled if true, the autosync is enabled
   * @throws StructureSynchronizerException if the incremental synchronization for this synchronizer cannot be enabled
   */
  void setAutosyncEnabled(@Nullable Long instanceId, boolean enabled) throws StructureSynchronizerException;

  /**
   * <p>Starts full synchronization in the specified direction.</p>
   *
   * @param instanceId the ID of the installed synchronizer
   * @param andEnable if true, incremental synchronization will be enabled right after the resync job is scheduled,
   * making sure that no changes will be missed
   * @param processId if not null, resync progress will be reported via this ID
   * @throws StructureSynchronizerException if resync could not be started
   */
  void resync(@Nullable Long instanceId, boolean andEnable, @Nullable Long processId)
    throws StructureSynchronizerException;

  /**
   * <p>Runs full resync without installing the synchronizer. (For example, this is used by Export and Import features.)</p>
   *
   * <p>Parameters of the appropriate type are required to run the synchronizer. Those are typically
   * collected from the user via synchronizer module form and created with {@link StructureSynchronizer#buildParametersFromForm}
   * or can be created by the custom code which is aware of the class of the synchronizer parameters.
   * Note that if {@link StructureSynchronizer#storeParameters(Object)} returns an empty String for {@code params}, then
   * the synchronizer will be called with {@code null} parameters.
   * </p>
   *
   * @param synchronizer the synchronizer to run
   * @param params synchronizer parameters
   * @param structureId the ID of the structure
   * @param user the user account under which the synchronization should run
   * @param processId if not null, resync progress will be reported via this ID
   * @throws StructureSynchronizerException if the synchronization could not be started
   */
  void resyncOnce(@NotNull StructureSynchronizer synchronizer, @Nullable Object params, @NotNull Long structureId, 
    @Nullable ApplicationUser user, @Nullable Long processId)
    throws StructureSynchronizerException;

  /**
   * Updates the parameters and/or the owner of an installed synchronizer. Incremental synchronization must be disabled
   * when calling this method.
   *
   * @param instanceId the ID of the installed synchronizer
   * @param params new synchronization parameters, of the type accepted by the synchronizer. If {@code null} or if 
   * {@link StructureSynchronizer#storeParameters(Object)} returns {@code null} or an empty String for the specified object,
   * the synchronizer will keep using its current parameters.
   * @param user the user account under which this synchronizer will run after the update. If {@code null},
   * the synchronizer will keep running under its current account
   * @return the new instance of the synchronizer if {@code user} is not {@code null} or {@code params} are not {@code null},
   * otherwise the existing instance
   * @throws StructureSynchronizerException if the synchronizer could not be updated
   */
  @NotNull
  SyncInstance updateSynchronizer(@Nullable Long instanceId, @Nullable Object params, @Nullable ApplicationUser user)
    throws StructureSynchronizerException;
}
