package com.almworks.jira.structure.api.settings;

import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.EnumMap;

/**
 * <p>This implementation of the {@link UISettings} can be used to modify the settings with
 * {@link StructureConfiguration#setUISettings}.</p>
 *
 * <p>This class is not thread safe, it should be a short-term object that is filled with
 * values, passed to <code>StructureConfiguration</code> and forgotten.</p>
 *
 * <p>In addition to implementing <code>UISettings</code>, this class has public setter methods
 * that allow to change the setting values.</p>
 *
 * @author Igor Sereda
 */
public class UISettingsBean implements UISettings {
  private final EnumMap<StructurePage, AutoSwitchStrategy> myAutoSwitchStrategyMap
    = new EnumMap<StructurePage, AutoSwitchStrategy>(StructurePage.class);

  private Boolean myAutoCollapseStructurePanel;

  private Boolean myKeepStructureWhileNavigating;

  /**
   * Creates an empty bean, with no settings set.
   */
  public UISettingsBean() {
  }

  /**
   * Creates a new bean and copies all set values from another instance of UISettings.
   *
   * @param copyFrom another instance of <code>UISettings</code>, <code>null</code> is ignored
   */
  public UISettingsBean(@Nullable UISettings copyFrom) {
    if (copyFrom != null) {
      for (StructurePage page : StructurePage.values()) {
        AutoSwitchStrategy strategy = copyFrom.getAutoSwitchStrategy(page);
        if (strategy != null) {
          myAutoSwitchStrategyMap.put(page, strategy);
        }
      }
      myAutoCollapseStructurePanel = copyFrom.getAutoCollapseStructurePanel();
    }
  }

  @Nullable
  public AutoSwitchStrategy getAutoSwitchStrategy(@NotNull StructurePage page) {
    return myAutoSwitchStrategyMap.get(page);
  }

  /**
   * Changes auto-switch strategy for the given page.
   *
   * @param page the page to change auto-switch for
   * @param strategy the new strategy, or <code>null</code> to un-set the option for that page
   * @return this instance
   * @see #getAutoSwitchStrategy
   */
  @NotNull
  public UISettingsBean setAutoSwitchStrategy(@NotNull StructurePage page, @Nullable AutoSwitchStrategy strategy) {
    if (page == null) throw new NullPointerException("page is null");
    if (strategy == null) {
      myAutoSwitchStrategyMap.remove(page);
    } else {
      myAutoSwitchStrategyMap.put(page, strategy);
    }
    return this;
  }

  public Boolean getAutoCollapseStructurePanel() {
    return myAutoCollapseStructurePanel;
  }

  public Boolean getKeepStructureWhileNavigating() {
    return myKeepStructureWhileNavigating;
  }

  public void setKeepStructureWhileNavigating(Boolean keepStructureWhileNavigating) {
    myKeepStructureWhileNavigating = keepStructureWhileNavigating;
  }

  public void setAutoCollapseStructurePanel(Boolean autoCollapseStructurePanel) {
    myAutoCollapseStructurePanel = autoCollapseStructurePanel;
  }

  public boolean isAnythingSet() {
    return !myAutoSwitchStrategyMap.isEmpty() || myAutoCollapseStructurePanel != null || myKeepStructureWhileNavigating != null;
  }
}
