package com.almworks.jira.structure.api.attribute.loader.builder;

import com.almworks.jira.structure.api.attribute.AttributeSpec;
import com.almworks.jira.structure.api.attribute.loader.*;
import com.almworks.jira.structure.api.item.ItemIdentity;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.*;

public abstract class AttributeLoaderBuilder<T, S extends AttributeLoaderBuilder<T, S>> {
  protected AttributeSpec<T> myAttributeSpec;
  protected AttributeCachingStrategy myCachingStrategy;
  protected TrailItemSet myGlobalTrail;
  private Set<AttributeContextDependency> myContextDependencies;

  public S spec(AttributeSpec<T> attributeSpec) {
    myAttributeSpec = attributeSpec;
    return self();
  }

  public S contextDependencies(@Nullable AttributeContextDependency... dependencies) {
    if (dependencies != null) {
      for (AttributeContextDependency dependency : dependencies) {
        contextDependency(dependency);
      }
    }
    return self();
  }

  public S contextDependency(@Nullable AttributeContextDependency dependency) {
    if (dependency != null) {
      Set<AttributeContextDependency> contextDependencies = myContextDependencies;
      if (contextDependencies == null) {
        myContextDependencies = contextDependencies = EnumSet.noneOf(AttributeContextDependency.class);
      }
      contextDependencies.add(dependency);
    }
    return self();
  }

  /**
   * Used to override default context dependencies.
   */
  public S noContextDependencies() {
    myContextDependencies = Collections.emptySet();
    return self();
  }

  public S cachingStrategy(AttributeCachingStrategy cachingStrategy) {
    myCachingStrategy = cachingStrategy;
    return self();
  }

  public S globalTrail(@Nullable TrailItemSet trail) {
    myGlobalTrail = trail == null ? myGlobalTrail : trail.union(myGlobalTrail);
    return self();
  }

  public S globalTrail(@Nullable ItemIdentity item) {
    return globalTrail(TrailItemSet.of(item));
  }

  protected S self() {
    //noinspection unchecked
    return (S) this;
  }

  @Nullable
  protected Set<AttributeContextDependency> buildContextDependencies() {
    Set<AttributeContextDependency> contextDependencies = myContextDependencies;
    if (contextDependencies == null) return null;
    if (contextDependencies.isEmpty()) return Collections.emptySet();
    return Collections.unmodifiableSet(EnumSet.copyOf(contextDependencies));
  }

  @NotNull
  protected static <R> R notNull(@Nullable R parameter, String parameterName) {
    if (parameter == null) {
      throw new IllegalArgumentException("null " + parameterName);
    }
    return parameter;
  }

  @Nullable
  protected static <R extends Collection> R nullableCollectionOfNonNulls(@Nullable R parameter, String parameterName) {
    if (parameter == null) {
      return parameter;
    }
    int i = 0;
    for (Object value : parameter) {
      if (value == null) {
        throw new IllegalArgumentException("null " + parameterName + "[" + i + "]");
      }
      i++;
    }
    return parameter;
  }
}
