package com.almworks.jira.structure.api.sync;

import com.almworks.jira.structure.api.util.La;
import com.almworks.jira.structure.api.util.MapObject;
import com.atlassian.jira.issue.Issue;
import com.atlassian.jira.issue.IssueConstant;
import com.atlassian.jira.issue.fields.CustomField;
import com.google.common.collect.ImmutableMap;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.*;

import static com.almworks.jira.structure.api.util.StructureUtil.getDebugIssueKey;
import static com.almworks.jira.structure.api.util.StructureUtil.nnv;

public class SyncAuditLogHelper {
  public static final La<Long, Map<String, Object>> ISSUE_INFO = new La<Long, Map<String, Object>>() {
    @Override
    public Map<String, Object> la(Long issueId) {
      return issueInfo(issueId);
    }
  };

  @NotNull
  public static Map<String, Object> issueInfo(@Nullable Issue issue) {
    if (issue == null) return Collections.emptyMap();
    return issueInfo(issue.getId(), issue.getKey());
  }

  @NotNull
  public static Map<String, Object> issueInfo(long issueId) {
    // todo is it acceptable to use getDebugIssueKey() for production purposes?
    // IS: what purposes?
    return issueInfo(issueId, getDebugIssueKey(issueId));
  }

  @NotNull
  public static Map<String, Object> issueInfo(long issueId, @Nullable String key) {
    ImmutableMap.Builder<String, Object> builder = ImmutableMap.builder();
    if (key != null) builder.put("key", key);
    builder.put("id", issueId);
    return builder.build();
  }
  
  @NotNull
  public static Map<String, Object> issueConstantInfo(IssueConstant ic) {
    if (ic == null) return Collections.emptyMap();
    ImmutableMap.Builder<String, Object> builder = ImmutableMap.builder();
    if (ic.getName() != null) builder.put("name", ic.getName());
    if (ic.getId() != null) builder.put("id", ic.getId());
    return builder.build();
  }

  @NotNull
  public static Map<String, Object> customFieldInfo(CustomField field) {
    if (field == null) return Collections.emptyMap();
    ImmutableMap.Builder<String, Object> builder = ImmutableMap.builder();
    if (field.getName() != null) builder.put("name", field.getName());
    if (field.getIdAsLong() != null) builder.put("id", field.getIdAsLong());
    return builder.build();
  }

  @NotNull
  public static MapObject createAuditLogEntryDescription(SyncInstance instance, IncrementalSyncData data, Map<String, Object> aux) {
    ImmutableMap.Builder<String, Object> map = ImmutableMap.builder();
    map.put("action", data == null ? "resync" : "sync");
    map.putAll(aux);
    map.put("ownerKey", instance.getUserKey());
    map.put("parameters", nnv(instance.getParameters(), Collections.emptyMap()));
    if (data != null) {
      map.put("jiraEvents", SyncEvents.JIRA_EVENT_TO_MAP.arrayList(data.getJiraEvents()));
      map.put("structureEvents", SyncEvents.STRUCTURE_EVENT_TO_MAP.arrayList(data.getStructureEvents()));
    }
    return new MapObject(map.build());
  }

  public static ImmutableMap<String, Object> getFailureDescription(@Nullable String reason, @Nullable Throwable throwable) {
    ImmutableMap.Builder<String, Object> builder = ImmutableMap.builder();
    builder.put("result", "failure");
    if (reason != null) {
      builder.put("reason", reason);
    }
    if (throwable != null) {
      builder.put("exception", getThrowableDescription(throwable));
    }
    return builder.build();
  }

  private static Map<String, Object> getThrowableDescription(Throwable throwable) {
    ImmutableMap.Builder<String, Object> throwableDescBuilder = ImmutableMap.builder();
    if (throwable.getMessage() != null) {
      throwableDescBuilder.put("message", throwable.getMessage());
    }
    throwableDescBuilder.
      put("class", throwable.getClass().getName()).
      put("stackTrace", Arrays.asList(throwable.getStackTrace()));
    if (throwable.getCause() != null) {
      throwableDescBuilder.put("cause", getThrowableDescription(throwable.getCause()));
    }
    return throwableDescBuilder.build();
  }

  public static boolean isFailed(MapObject action) {
    return "failure".equals(action.getString("result"));
  }
}
