package com.almworks.jira.structure.api.settings;

import com.almworks.jira.structure.api.attribute.AttributeSpec;
import org.jetbrains.annotations.NotNull;

import java.util.Set;

/**
 * <p>Attribute sensitivity settings govern which attributes will be considered sensitive when calculating a multi-row aggregate value (by
 * an aggregate, a propagate, or a scanning loader). This plays a role when a user does not have view access to some of the items in the forest,
 * yet they look at a total or some other aggregate value.</p>
 *
 * <p>As an example let's consider aggregate attribute Total Story Points, calculating over a folder, which contains issues X, Y, and Z.
 * The current user does not have access to Z, so they only see the folder, X, and Y.</p>
 *
 * <p>If the aggregated attribute - Story Points - is <strong>sensitive</strong>, then the value for the folder will be calculated only over those
 * issues that the user can actually observe, in this case, X and Y. Another user, who has access to all issues, will see a total of X, Y, and Z -
 * potentially a different number.</p>
 *
 * <p>If the aggregated attribute is <strong>non-sensitive</strong>, the total for the folder will show the same number for all users, even if
 * they don't have access to some of the sub-items. Note that the inaccessible item is not disclosed to the user in any case.</p>
 *
 * <p>These settings allow Jira admins to set the default mode (see {@link #getSensitivityMode()}) and exceptions (see {@link #getNonSensitiveAttributes()}).
 * </p>
 *
 * @see StructureConfiguration#getAttributeSensitivitySettings
 * @see StructureConfiguration#setAttributeSensitivitySettings
 * @see ImmutableAttributeSensitivitySettingsBean
 */
public interface AttributeSensitivitySettings {
  /**
   * Returns the default sensitivity mode.
   *
   * @return sensitivity mode
   */
  @NotNull
  AttributeSensitivityMode getSensitivityMode();

  /**
   * <p>Returns attributes that were explicitly set to be non-sensitive, regardless of the default mode.</p>
   *
   * <p>Note that if the mode is {@link AttributeSensitivityMode#PERMISSIVE}, this method may still return some attributes, although
   * that will not have any additional effect - everything is non-sensitive by mode.</p>
   *
   * @return a list of attributes set to be non-sensitive
   */
  @NotNull
  Set<AttributeSpec<?>> getNonSensitiveAttributes();
}
