package com.almworks.jira.structure.api.attribute.loader;

import com.almworks.jira.structure.api.attribute.AttributeValue;
import com.almworks.jira.structure.api.row.StructureRow;
import com.atlassian.annotations.PublicSpi;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.List;

/**
 * <p>Aggregate attributes have values calculated up through the hierarchy. The parent row's value depends on the
 * values of its children. For example, a total time spent for all issues under a folder is an aggregate.</p>
 *
 * <p>Aggregate attribute loader represents an aggregate attribute. Aggregate loading function for a row {@code R}
 * can calculate values based on:</p>
 * <ul>
 * <li>Already calculated values of the same attribute for direct children rows of R;</li>
 * <li>Values of attribute dependencies for the row R (but not its children);</li>
 * <li>{@link StructureRow} data for R and its children;</li>
 * <li>Context values, if declared with {@link #getContextDependencies}.</li>
 * </ul>
 *
 * <p><strong>The loader must be a stable function.</strong> It must not depend on any other inputs, and should
 * provide the same result for the same inputs every time: {@code V(row) = AGGREGATE([V(children(row))], dependencies(row))}.</p>
 *
 * <p>The values for children rows and the dependencies for the current row are guaranteed to be loaded by the time the loading function
 * is called.</p>
 *
 * <h3>Using loader data</h3>
 *
 * <p>Note that sometimes an aggregate loader will need to store more data in {@link AttributeValue} than the resulting Java
 * type allows. In that case, use loader data - see {@link AttributeValue#withData}.</p>
 *
 * <h3>Sensitive data control</h3>
 *
 * <p>An aggregate, as well as all multi-row loaders, is potentially capable of accessing data calculated for
 * rows that the requesting user is not allowed to see. For example, a total budget may be calculated over a
 * number of sub-budgets, and a user may not have access to all of them.</p>
 *
 * <p>If the aggregated attribute is
 * considered sensitive (according to {@link  com.almworks.jira.structure.api.settings.StructureConfiguration#getAttributeSensitivitySettings()}), then
 * the calls to {@link AttributeLoaderContext#getDependencyValue} will not disclose the hidden value. Instead,
 * the loader function will receive {@link AttributeValue#inaccessible()}.</p>
 *
 * <p>In other words, you should not worry about implementing security checks in an aggregate, the system does
 * it for you.</p>
 *
 * <h3>Super-root support</h3>
 *
 * <p>An aggregate may be called to calculate a value for a super-root row (see {@link SuperRootRow}),
 * passing all root values as children values.</p>
 *
 * @param <T> type of the loaded value
 *
 * @see AttributeLoader
 * @see PropagateAttributeLoader
 * @see ScanningAttributeLoader
 * @see com.almworks.jira.structure.api.settings.AttributeSensitivitySettings
 */
@PublicSpi
public interface AggregateAttributeLoader<T> extends MultiRowAttributeLoader<T> {
  /**
   * <p>Performs aggregate loading.</p>
   *
   * <p>If the value is missing or empty, the loader should return {@link AttributeValue#undefined()}. If for some reason this loader
   * is not applicable for the given row, it should return {@code null}. (Note, however, that if multiple aggregate loaders work on the same
   * attribute, it could be tricky and lead to unexpected results.)</p>
   *
   * <p>Note: children values may contain loop markers</p>
   *
   * @param childrenValues the values of this attribute for all children rows, in the same order as the children rows
   * @param context loading context
   * @return the calculated value
   */
  @Nullable
  AttributeValue<T> loadValue(@NotNull List<AttributeValue<T>> childrenValues, @NotNull AggregateAttributeContext context);
}
