package com.almworks.jira.structure.api.attribute;

import com.almworks.integers.LongList;
import com.almworks.jira.structure.api.forest.raw.Forest;
import com.almworks.jira.structure.api.item.ItemTracker;
import com.almworks.jira.structure.api.pull.DataVersion;
import com.atlassian.annotations.PublicApi;
import org.jetbrains.annotations.NotNull;

import java.util.Collection;
import java.util.Collections;

/**
 * <p>{@code ConsistentRowValues} carries the values loaded in a consistent way, that is, when all the requested data retained their values
 * during the loading process.</p>
 *
 * <p>Note that these objects also contain the specific version of {@link Forest} that was used to calculate the values. This may be important
 * when the consistent values are applied - it's better to use the forest retrieved here from {@link #getForest()}, rather than request it separately,
 * because it might change and the values contained in this class may not match the updated forest.</p>
 *
 * @see StructureAttributeService#getConsistentAttributeValues
 */
@PublicApi
public class ConsistentRowValues {
  public static final ConsistentRowValues EMPTY = new ConsistentRowValues(Forest.EMPTY, LongList.EMPTY, Collections.emptySet(), RowValues.EMPTY,
    DataVersion.ZERO, DataVersion.ZERO);

  @NotNull
  private final Forest myForest;

  @NotNull
  private final LongList myRequestedRows;

  @NotNull
  private final Collection<? extends AttributeSpec<?>> myRequestedAttributes;

  @NotNull
  private final RowValues myValues;

  @NotNull
  private final DataVersion myForestVersion;

  @NotNull
  private final DataVersion myItemsVersion;

  public ConsistentRowValues(@NotNull Forest forest, @NotNull LongList requestedRows,
    @NotNull Collection<? extends AttributeSpec<?>> requestedAttributes, @NotNull RowValues values,
    @NotNull DataVersion forestVersion, @NotNull DataVersion itemsVersion)
  {
    myForest = forest;
    myRequestedRows = requestedRows;
    myRequestedAttributes = requestedAttributes;
    myValues = values;
    myForestVersion = forestVersion;
    myItemsVersion = itemsVersion;
  }

  /**
   * Returns the values for the requested rows and attributes.
   *
   * @return values
   */
  @NotNull
  public RowValues getValues() {
    return myValues;
  }

  /**
   * Returns the forest that was used to calculate the consistent values.
   *
   * @return forest
   */
  @NotNull
  public Forest getForest() {
    return myForest;
  }

  /**
   * Returns the rows that were requested from {@link StructureAttributeService#getConsistentAttributeValues}. Note that the included values
   * may contain values for other rows, but they are not guaranteed to be consistent.
   *
   * @return requested rows
   */
  @NotNull
  public LongList getRequestedRows() {
    return myRequestedRows;
  }

  /**
   * Returns the attributes that were requested from {@link StructureAttributeService#getConsistentAttributeValues}. Note that the included
   * values may contain values for other attributes, but they are not guaranteed to be consistent.
   *
   * @return requested attributes
   */
  @NotNull
  public Collection<? extends AttributeSpec<?>> getRequestedAttributes() {
    return myRequestedAttributes;
  }

  /**
   * Returns the version of forest that was used to calculate the values.
   *
   * @return forest version
   */
  @NotNull
  public DataVersion getForestVersion() {
    return myForestVersion;
  }

  /**
   * Returns the version of the item update stream that was current when the values were loaded.
   *
   * @return item stream version
   * @see ItemTracker
   */
  @NotNull
  public DataVersion getItemsVersion() {
    return myItemsVersion;
  }
}
