package com.almworks.jira.structure.api.settings;

import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

/**
 * <p>Provides access to the user interface settings of the Structure Widget. These settings
 * govern various aspects of Structure Widget configuration and behavior.</p>
 *
 * <p>There's a global default settings collection and they can be tweaked per-user and
 * possibly per-project. See {@link StructureConfiguration#setUISettings}
 * for mode detail.</p>
 *
 * <p>A <code>null</code> value is treated as "not set". In most cases when an instance of <code>UISettings</code>
 * is retrieved from <code>StructureConfiguration</code>, all possible settings will have a value.
 * "Not set" makes more sense when updating settings, where only the specified settings will be updated.</p>
 *
 * @see StructureConfiguration
 *
 * @author Igor Sereda
 */
public interface UISettings {
  /**
   * Returns auto-switch strategy for the specified page. Auto-switch strategy defines
   * which structure to load at the initial page load.
   *
   * @param page structure page that is being loaded, currently only {@link StructurePage#ISSUE_VIEW},
   * {@link StructurePage#PROJECT_TAB}, {@link StructurePage#COMPONENT_TAB} and {@link StructurePage#VERSION_TAB}
   * make sense.
   * @return auto-switch strategy for the specified page, or <code>null</code> if not set
   */
  @Nullable
  AutoSwitchStrategy getAutoSwitchStrategy(@NotNull StructurePage page);

  /**
   * Returns true if Structure panel on the issue page should automatically collapse/minimize when the displayed
   * issue is not present in the initially selected structure.
   *
   * @return true if structure panel auto-collapse is enabled, or {@code null} if the settings is unknown
   */
  @Nullable
  Boolean getAutoCollapseStructurePanel();

  /**
   * <p>Returns true if the structure selection should be kept when the user clicks on an issue in the Structure widget.</p>
   *
   * <p>This allows the user to go from one issue to another, looking at the Structure panel, and the structure displayed
   * will remain the same. If the user opens an issue page by any other means, the structure will be selected according
   * to other settings.</p>
   *
   * <p>The default value for this setting (in case {@code null} is returned) is {@code true}.</p>
   *
   * @return true if the structure selection should be kept when opening an issue page through the Structure widget
   */
  @Nullable
  Boolean getKeepStructureWhileNavigating();

  /**
   * Utility method that is used to check whether this instance does not have any settings set
   * at all.
   *
   * @return true if at least one setting is defined
   */
  boolean isAnythingSet();
}
