package com.almworks.jira.structure.api.rest;

import com.almworks.jira.structure.api.attribute.AttributeSpec;
import com.almworks.jira.structure.api.attribute.ValueFormat;
import com.almworks.jira.structure.api.util.ConsiderateLogger;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlRootElement;
import java.util.*;
import java.util.stream.Collectors;

import static com.almworks.jira.structure.api.attribute.ValueFormat.*;

@XmlRootElement
public final class RestAttributeSpec {
  private static final Logger logger = LoggerFactory.getLogger(AttributeSpec.class);
  private static final ConsiderateLogger considerateLogger = new ConsiderateLogger(logger);
  
  private static final Map<String, ValueFormat<?>> RECOGNIZED_FORMATS = createRecognizedFormatsMap();
  /**
   * Universal attribute ID
   */
  @XmlElement
  public String id;

  /**
   * (Optional) Arbitrary parameters, interpreted by the attribute provider (any JSON)
   */
  @XmlElement
  public Map params;

  /**
   * Format - must be one of {@link com.almworks.jira.structure.api.attribute.ValueFormat} IDs
   * recognizable by the REST resource. (Custom value formats are not supported.)
   */
  @XmlElement
  public String format;

  private static Map<String, ValueFormat<?>> createRecognizedFormatsMap() {
    HashMap<String, ValueFormat<?>> r = new HashMap<>();
    ValueFormat[] formats = {
      HTML, TEXT, JSON_OBJECT, JSON_ARRAY, NUMBER, BOOLEAN, ID, TIME, DURATION, ANY
    };
    for (ValueFormat format : formats) {
      r.put(format.getFormatId(), format);
    }
    return Collections.unmodifiableMap(r);
  }

  @Nullable
  public static AttributeSpec<?> fromRest(@Nullable RestAttributeSpec spec) {
    if (spec == null) return null;
    try {
      return fromRestNotNull(spec);
    } catch (IllegalArgumentException e) {
      considerateLogger.warn("spec " + spec, "cannot be deserialized: " + e.getMessage());
      return null;
    }
  }

  @NotNull
  public static AttributeSpec<?> fromRestNotNull(@NotNull RestAttributeSpec spec) throws IllegalArgumentException {
    //noinspection ConstantConditions
    if (spec == null) throw new IllegalArgumentException();
    ValueFormat<?> format = RECOGNIZED_FORMATS.get(spec.format);
    if (format == null) {
      throw new IllegalArgumentException("format " + spec.format + " is unknown");
    }
    //noinspection unchecked
    return new AttributeSpec<>(spec.id, format, spec.params);
  }

  @NotNull
  public static RestAttributeSpec toRest(@NotNull AttributeSpec<?> spec) {
    RestAttributeSpec ras = new RestAttributeSpec();
    ras.id = spec.getId();
    ras.format = spec.getFormat().getFormatId();
    ras.params = spec.getParamsMap();
    if (ras.params.isEmpty()) ras.params = null;
    return ras;
  }

  @NotNull
  public static List<AttributeSpec<?>> fromRestCollection(@Nullable Collection<? extends RestAttributeSpec> source) {
    if (source == null) return Collections.emptyList();
    return source.stream().map(RestAttributeSpec::fromRest).filter(Objects::nonNull).collect(Collectors.toList());
  }

  @Override
  public String toString() {
    return "(" + id + "," + format + "," + params + ")";
  }
}
