package com.almworks.jira.structure.api.attribute.loader.builder;

import com.almworks.jira.structure.api.attribute.AttributeValue;
import com.almworks.jira.structure.api.attribute.loader.*;
import com.almworks.jira.structure.api.item.ItemIdentity;
import org.jetbrains.annotations.Nullable;

import java.util.*;
import java.util.function.*;

public class ItemAttributeLoaderBuilder<T, I> extends ArbitraryDependenciesAttributeLoaderBuilder<T, ItemAttributeLoaderBuilder<T, I>> {
  private AbstractLoadingStrategy<T, I> myLoadingStrategy;
  private Predicate<String> myItemTypeSupportedPredicate;
  private Class<I> myItemClass;
  private Function<I, ItemIdentity> myItemTrailFunction;
  private Function<I, TrailItemSet> myTrailItemSetFunction;
  private Function<T, ItemIdentity> myValueBasedTrailFunction;
  private Boolean myYieldOnNull;
  private BiConsumer<Collection<ItemIdentity>, AttributeContext> myPreloadFunction;

  public ItemAttributeLoaderBuilder<T, I> valueFunctionIIAV(BiFunction<ItemIdentity, ItemAttributeContext, AttributeValue<T>> loadingFunction) {
    myLoadingStrategy = new SimpleLoadStrategy<>(loadingFunction);
    return this;
  }

  public ItemAttributeLoaderBuilder<T, I> valueFunctionIIAV(Function<ItemIdentity, AttributeValue<T>> valueFunction) {
    return valueFunctionIIAV((item, ctx) -> valueFunction.apply(item));
  }

  public ItemAttributeLoaderBuilder<T, I> valueFunctionAV(BiFunction<? super I, ItemAttributeContext, AttributeValue<T>> itemLoadingFunction) {
    myLoadingStrategy = new ResolveAndLoadStrategy<>(itemLoadingFunction);
    return this;
  }

  public ItemAttributeLoaderBuilder<T, I> valueFunctionAV(Function<? super I, AttributeValue<T>> itemFunction) {
    return valueFunctionAV((item, ctx) -> itemFunction.apply(item));
  }

  public ItemAttributeLoaderBuilder<T, I> valueFunctionII(BiFunction<ItemIdentity, ItemAttributeContext, ? extends T> valueFunction) {
    myLoadingStrategy = new LoadAndHandleStrategy<>(valueFunction);
    return self();
  }

  public ItemAttributeLoaderBuilder<T, I> valueFunctionII(Function<ItemIdentity, ? extends T> valueFunction) {
    return valueFunctionII((item, ctx) -> valueFunction.apply(item));
  }

  public ItemAttributeLoaderBuilder<T, I> valueFunction(BiFunction<? super I, ItemAttributeContext, ? extends T> itemFunction) {
    myLoadingStrategy = new ResolveAndLoadAndHandleStrategy<>(itemFunction);
    return self();
  }

  public ItemAttributeLoaderBuilder<T, I> valueFunction(Function<? super I, ? extends T> itemFunction) {
    return valueFunction((item, ctx) -> itemFunction.apply(item));
  }

  public <C> ItemAttributeLoaderBuilder<T, C> itemClass(Class<C> itemClass) {
    ItemAttributeLoaderBuilder<T, C> r = (ItemAttributeLoaderBuilder<T, C>) this;
    r.myItemClass = notNull(itemClass, "itemClass");
    return r;
  }

  public ItemAttributeLoaderBuilder<T, I> itemType(String... itemTypes) {
    if (itemTypes.length == 0) {
      throw new IllegalArgumentException("need at least one item type");
    }
    Set<String> acceptableTypes = new HashSet<>(itemTypes.length);
    for (String itemType : itemTypes) {
      if (itemType == null) {
        throw new IllegalArgumentException("itemType cannot be null");
      }
      acceptableTypes.add(itemType);
    }
    if (acceptableTypes.size() == 1) {
      String itemType = acceptableTypes.iterator().next();
      myItemTypeSupportedPredicate = itemType::equals;
    } else {
      myItemTypeSupportedPredicate = acceptableTypes::contains;
    }
    return this;
  }

  public ItemAttributeLoaderBuilder<T, I> anyItemType() {
    myItemTypeSupportedPredicate = null;
    return this;
  }

  public ItemAttributeLoaderBuilder<T, I> itemTrail(Function<I, ItemIdentity> itemTrailFunction) {
    myItemTrailFunction = itemTrailFunction;
    if (itemTrailFunction != null) contextDependency(AttributeContextDependency.TRAIL);
    return self();
  }

  public ItemAttributeLoaderBuilder<T, I> trailItemSet(Function<I, TrailItemSet> trailItemSetFunction) {
    myTrailItemSetFunction = trailItemSetFunction;
    if (trailItemSetFunction != null) contextDependency(AttributeContextDependency.TRAIL);
    return self();
  }

  public ItemAttributeLoaderBuilder<T, I> valueTrail(Function<T, ItemIdentity> valueBasedTrailFunction) {
    myValueBasedTrailFunction = valueBasedTrailFunction;
    if (valueBasedTrailFunction != null) contextDependency(AttributeContextDependency.TRAIL);
    return self();
  }

  public ItemAttributeLoaderBuilder<T, I> yieldOnNull() {
    myYieldOnNull = true;
    return self();
  }

  public ItemAttributeLoaderBuilder<T, I> preload(BiConsumer<Collection<ItemIdentity>, AttributeContext> preloadFunction) {
    myPreloadFunction = preloadFunction;
    return self();
  }


  public ItemAttributeLoader<T> build() {
    AbstractLoadingStrategy<T, I> strategy = notNull(myLoadingStrategy, "loading function");
    if (myItemClass != null) {
      strategy.setItemClass(myItemClass);
    }
    if (myYieldOnNull != null) {
      strategy.setYieldOnNull(myYieldOnNull);
    }
    if (myValueBasedTrailFunction != null) {
      strategy.setValueBasedTrailFunction(myValueBasedTrailFunction);
    }
    if (myItemTrailFunction != null) {
      strategy.setItemTrailFunction(myItemTrailFunction);
    }
    if (myTrailItemSetFunction != null) {
      strategy.setTrailItemSetFunction(myTrailItemSetFunction);
    }

    return new BaseItemAttributeLoader<>(
      notNull(myAttributeSpec, "attributeSpec"),
      nullableCollectionOfNonNulls(buildDependencies(), "dependencies"),
      nullableCollectionOfNonNulls(buildContextDependencies(), "contextDependencies"),
      myCachingStrategy,
      myGlobalTrail,
      myItemTypeSupportedPredicate,
      strategy.createLoadingFunction(),
      myPreloadFunction
    );
  }


  private static <I> void applyItemTrailFunction(I item, Function<I, ItemIdentity> itemTrailFunction, ItemAttributeContext ctx) {
    if (itemTrailFunction != null) {
      ItemIdentity trail = itemTrailFunction.apply(item);
      if (trail != null) {
        ctx.addTrail(trail);
      }
    }
  }

  private static <I> void applyTrailItemSetFunction(I item, Function<I, TrailItemSet> trailItemSetFunction, ItemAttributeContext ctx) {
    if (trailItemSetFunction != null) {
      TrailItemSet trail = trailItemSetFunction.apply(item);
      if (trail != null) {
        ctx.addTrail(trail);
      }
    }
  }

  @Nullable
  private static <T> AttributeValue<T> handleValue(T value, boolean yieldOnNull, Function<? super T, ItemIdentity> valueBasedTrailFunction,
    ItemAttributeContext context)
  {
    if (value == null && yieldOnNull) return null;
    if (value != null && valueBasedTrailFunction != null) {
      ItemIdentity trailItem = valueBasedTrailFunction.apply(value);
      if (trailItem != null) {
        context.addTrail(trailItem);
      }
    }
    return AttributeValue.ofNullable(value);
  }


  private static abstract class AbstractLoadingStrategy<T, I> {
    abstract BiFunction<ItemIdentity, ItemAttributeContext, AttributeValue<T>> createLoadingFunction();

    public void setValueBasedTrailFunction(Function<? super T, ItemIdentity> valueBasedTrailFunction) {
      throw new IllegalArgumentException(getClass() + " does not accept valueBasedTrailFunction");
    }

    public void setYieldOnNull(boolean yieldOnNull) {
      throw new IllegalArgumentException(getClass() + " does not accept yieldOnNull");
    }

    public void setItemClass(Class<I> itemClass) {
      throw new IllegalArgumentException(getClass() + " does not accept typed");
    }

    public void setItemTrailFunction(Function<I, ItemIdentity> itemTrailFunction) {
      throw new IllegalArgumentException(getClass() + " does not accept itemTrailFunction");
    }

    public void setTrailItemSetFunction(Function<I, TrailItemSet> trailItemSetFunction) {
      throw new IllegalArgumentException(getClass() + " does not accept trailItemSetFunction");
    }
  }


  private static class SimpleLoadStrategy<T, I> extends AbstractLoadingStrategy<T, I> {
    private final BiFunction<ItemIdentity, ItemAttributeContext, AttributeValue<T>> myLoadingFunction;

    SimpleLoadStrategy(BiFunction<ItemIdentity, ItemAttributeContext, AttributeValue<T>> loadingFunction) {
      myLoadingFunction = notNull(loadingFunction, "loadingFunction");
    }

    @Override
    BiFunction<ItemIdentity, ItemAttributeContext, AttributeValue<T>> createLoadingFunction() {
      return myLoadingFunction;
    }
  }


  private static class ResolveAndLoadStrategy<T, I> extends AbstractLoadingStrategy<T, I> {
    private BiFunction<? super I, ItemAttributeContext, AttributeValue<T>> myItemLoadingFunction;
    private Class<I> myItemClass;
    private Function<I, ItemIdentity> myItemTrailFunction;

    ResolveAndLoadStrategy(BiFunction<? super I, ItemAttributeContext, AttributeValue<T>> itemLoadingFunction) {
      myItemLoadingFunction = notNull(itemLoadingFunction, "itemLoadingFunction");
    }

    @Override
    public void setItemClass(Class<I> itemClass) {
      myItemClass = itemClass;
    }

    @Override
    public void setItemTrailFunction(Function<I, ItemIdentity> itemTrailFunction) {
      myItemTrailFunction = itemTrailFunction;
    }

    @Override
    BiFunction<ItemIdentity, ItemAttributeContext, AttributeValue<T>> createLoadingFunction() {
      return new ResolveAndLoad<>(notNull(myItemClass, "itemClass"), myItemLoadingFunction, myItemTrailFunction);
    }
  }


  private static class ResolveAndLoad<T, I> implements BiFunction<ItemIdentity, ItemAttributeContext, AttributeValue<T>> {
    private final Class<I> myItemClass;
    private final BiFunction<? super I, ItemAttributeContext, AttributeValue<T>> myItemLoadingFunction;
    private final Function<I, ItemIdentity> myItemTrailFunction;

    ResolveAndLoad(Class<I> itemClass, BiFunction<? super I, ItemAttributeContext, AttributeValue<T>> itemLoadingFunction,
      Function<I, ItemIdentity> itemTrailFunction)
    {
      myItemClass = itemClass;
      myItemLoadingFunction = itemLoadingFunction;
      myItemTrailFunction = itemTrailFunction;
    }

    @Override
    public AttributeValue<T> apply(ItemIdentity itemId, ItemAttributeContext ctx) {
      I item = ctx.getItem(myItemClass);
      if (item == null) {
        return AttributeValue.undefined();
      }
      applyItemTrailFunction(item, myItemTrailFunction, ctx);
      return myItemLoadingFunction.apply(item, ctx);
    }
  }


  private static abstract class HandleStrategy<T, I> extends AbstractLoadingStrategy<T, I> {
    protected boolean myYieldOnNull;
    protected Function<? super T, ItemIdentity> myValueBasedTrailFunction;

    @Override
    public void setYieldOnNull(boolean yieldOnNull) {
      myYieldOnNull = yieldOnNull;
    }

    @Override
    public void setValueBasedTrailFunction(Function<? super T, ItemIdentity> valueBasedTrailFunction) {
      myValueBasedTrailFunction = valueBasedTrailFunction;
    }
  }


  private static class LoadAndHandleStrategy<T, I> extends HandleStrategy<T, I> {
    private BiFunction<ItemIdentity, ItemAttributeContext, ? extends T> myValueFunction;

    LoadAndHandleStrategy(BiFunction<ItemIdentity, ItemAttributeContext, ? extends T> valueFunction) {
      myValueFunction = valueFunction;
    }

    @Override
    BiFunction<ItemIdentity, ItemAttributeContext, AttributeValue<T>> createLoadingFunction() {
      return new LoadAndHandle<>(notNull(myValueFunction, "valueFunction"), myYieldOnNull, myValueBasedTrailFunction);
    }
  }

  private static class LoadAndHandle<T> implements BiFunction<ItemIdentity, ItemAttributeContext, AttributeValue<T>> {
    private final boolean myYieldOnNull;
    private final Function<? super T, ItemIdentity> myValueBasedTrailFunction;
    private final BiFunction<ItemIdentity, ItemAttributeContext, ? extends T> myValueFunction;

    LoadAndHandle(BiFunction<ItemIdentity, ItemAttributeContext, ? extends T> valueFunction, boolean yieldOnNull,
      Function<? super T, ItemIdentity> valueBasedTrailFunction)
    {
      myValueFunction = valueFunction;
      myYieldOnNull = yieldOnNull;
      myValueBasedTrailFunction = valueBasedTrailFunction;
    }

    @Override
    public AttributeValue<T> apply(ItemIdentity itemId, ItemAttributeContext context) {
      T value = myValueFunction.apply(itemId, context);
      return handleValue(value, myYieldOnNull, myValueBasedTrailFunction, context);
    }
  }


  private static class ResolveAndLoadAndHandleStrategy<T, I> extends HandleStrategy<T, I> {
    private BiFunction<? super I, ItemAttributeContext, ? extends T> myItemFunction;
    private Class<I> myItemClass;
    private Function<I, ItemIdentity> myItemTrailFunction;
    private Function<I, TrailItemSet> myTrailItemSetFunction;

    ResolveAndLoadAndHandleStrategy(BiFunction<? super I, ItemAttributeContext, ? extends T> itemFunction) {
      myItemFunction = itemFunction;
    }

    @Override
    public void setItemClass(Class<I> itemClass) {
      myItemClass = itemClass;
    }

    @Override
    public void setItemTrailFunction(Function<I, ItemIdentity> itemTrailFunction) {
      myItemTrailFunction = itemTrailFunction;
    }

    @Override
    public void setTrailItemSetFunction(Function<I, TrailItemSet> trailItemSetFunction) {
      myTrailItemSetFunction = trailItemSetFunction;
    }

    @Override
    BiFunction<ItemIdentity, ItemAttributeContext, AttributeValue<T>> createLoadingFunction() {
      return new ResolveAndLoadAndHandle<T, I>(notNull(myItemFunction, "itemFunction"), myYieldOnNull, myValueBasedTrailFunction,
        notNull(myItemClass, "itemClass"), myItemTrailFunction, myTrailItemSetFunction);
    }
  }


  private static class ResolveAndLoadAndHandle<T, I> implements BiFunction<ItemIdentity, ItemAttributeContext, AttributeValue<T>> {
    private final boolean myYieldOnNull;
    private final Function<? super T, ItemIdentity> myValueBasedTrailFunction;
    private final BiFunction<? super I, ItemAttributeContext, ? extends T> myItemFunction;
    private final Class<I> myItemClass;
    private final Function<I, ItemIdentity> myItemTrailFunction;
    private final Function<I, TrailItemSet> myTrailItemSetFunction;

    ResolveAndLoadAndHandle(BiFunction<? super I, ItemAttributeContext, ? extends T> itemFunction, boolean yieldOnNull,
      Function<? super T, ItemIdentity> valueBasedTrailFunction, Class<I> itemClass, Function<I, ItemIdentity> itemTrailFunction,
      Function<I, TrailItemSet> trailItemSetFunction)
    {
      myItemFunction = itemFunction;
      myYieldOnNull = yieldOnNull;
      myValueBasedTrailFunction = valueBasedTrailFunction;
      myItemClass = itemClass;
      myItemTrailFunction = itemTrailFunction;
      myTrailItemSetFunction = trailItemSetFunction;
    }

    @Override
    public AttributeValue<T> apply(ItemIdentity itemId, ItemAttributeContext context) {
      I item = context.getItem(myItemClass);
      if (item == null) {
        return AttributeValue.undefined();
      }
      applyItemTrailFunction(item, myItemTrailFunction, context);
      applyTrailItemSetFunction(item, myTrailItemSetFunction, context);
      T value = myItemFunction.apply(item, context);
      return handleValue(value, myYieldOnNull, myValueBasedTrailFunction, context);
    }
  }
}
