package com.almworks.jira.structure.api.view;

import com.almworks.jira.structure.api.auth.StructureAuth;
import com.almworks.jira.structure.api.error.StructureException;
import com.almworks.jira.structure.api.permissions.PermissionLevel;
import com.almworks.jira.structure.api.settings.StructurePage;
import com.almworks.jira.structure.api.structure.Structure;
import com.almworks.jira.structure.api.structure.StructureManager;
import com.atlassian.annotations.PublicApi;
import com.atlassian.jira.user.ApplicationUser;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.List;

/**
 * <p>{@code StructureViewManager} is a component that manages Structure views.</p>
 *
 * <p>A view is a collection of visual configuration parameters that define how Structure grid
 * looks like. <em>(Initially, a view defines only the columns used in the grid, but we intend to expand
 * the notion of view in the future versions.)</em></p>
 *
 * <p>Managing views and structures have a lot in common. This service provides methods for
 * retrieving and updating views, like {@link StructureManager} does for structures.
 * {@link StructureView}, the interface that represents a view, is very similar to the {@link Structure} interface.</p>
 *
 * <p>Besides views, this service also manages view settings for individual structures, as well as
 * global view settings. View settings define which views are offered in the drop-down menus on
 * different pages with Structure grid, and which views are used by default.</p>
 *
 * <p>Unless otherwise stated, all methods are thread-safe.
 * Note that the methods of the {@link StructureView} interface are not thread-safe - you need
 * to retrieve an instance of {@code StructureView}, do the work with it in one thread, and forget it.</p>
 *
 * <p>All methods can accept {@code null} arguments for convenience. {@code null} structure ID or view ID means
 * the same as an ID of a missing structure or view.</p>
 *
 * <p>All methods operate in the authentication context as defined by {@link StructureAuth}. By default,
 * this context coincides with the JIRA authentication context.
 *
 * @see StructureView
 * @see ViewSettings
 * @author Igor Sereda
 */
@PublicApi
public interface StructureViewManager {
  /**
   * <p>Retrieves a view specified by the numeric view ID and checks if
   * the {@link StructureAuth#getUser() current user} has the specified access level for that view.
   *
   * <p>Unless security checks are {@link StructureAuth#isSecurityOverridden() disabled},
   * the user is also checked for being allowed to use Structure plugin at all.</p>
   *
   * @param viewId the ID of the view
   * @param requiredLevel when checking for user's permission, require that the user has at least the specified access level.
   * Passing null or {@link PermissionLevel#NONE} effectively disables view permissions check, but the user will be checked for
   * being allowed to use Structure plugin.
   * @return an instance of the view, not null
   * @throws StructureException if the view is not found, or the user is not allowed to use Structure plugin,
   * or the user does not have the required access level to this view, or if any other problem is encountered
   */
  @NotNull
  StructureView getView(@Nullable Long viewId, @Nullable PermissionLevel requiredLevel) throws StructureException;

  /**
   * <p>Retrieves a list of all views that the {@link StructureAuth#getUser() current user}
   * has the specified access level to.</p>
   *
   * <p>This method never throws an exception, but it may return an empty list if the user does not have
   * access to any view.</p>
   *
   * @param requiredLevel when checking for user's permission, require that the user has at least the specified permission for every
   * view in the returned list.
   * Passing null or {@link PermissionLevel#NONE} effectively disables the view permissions check, but the user will be checked for
   * being allowed to use Structure plugin.
   * @return a list of views available to the user, not null
   */
  @NotNull
  List<StructureView> getViews(@Nullable PermissionLevel requiredLevel);

  /**
   * Calculates the access level that the specified user has to the specified view.
   * Note that {@link StructureAuth the authentication context} doesn't influence this method.
   *
   * @param viewId the ID of the view
   * @param user the user, or null for anonymous
   * @return the access level. If the view does not exist or {@code viewId} is {@code null},
   * the result is {@link PermissionLevel#NONE}
   */
  @NotNull
  PermissionLevel getViewPermission(@Nullable Long viewId, @Nullable ApplicationUser user);

  /**
   * <p>Checks if the specified view exists and the {@link StructureAuth#getUser() current user} has the given 
   * access level to it.</p>
   *
   * <p>If security checks are {@link StructureAuth#isSecurityOverridden() disabled},
   * the method only checks that the specified view exists.</p>
   *
   * @param viewId the ID of the view
   * @param level required access level. Passing null or {@link PermissionLevel#NONE} effectively disables 
   *              the view permissions check, but the user will be checked for being allowed to use Structure plugin.
   * @return {@code true} if the specified view is accessible
   */
  boolean isAccessible(@Nullable Long viewId, @Nullable PermissionLevel level);

  /**
   * Creates an empty new view. The view returned is not yet persisted - you need to call the required
   * setter methods and then call {@link StructureView#saveChanges}
   * to write the new view to the database.
   *
   * @return the instance of the new view, not persisted to the database
   * @see StructureView#saveChanges
   */
  @NotNull
  StructureView createView();

  /**
   * <p>Deletes a view.</p>
   *
   * <p>Note that when a view is deleted, it's not removed from the {@link ViewSettings} automatically,
   * but that must not be a problem since every time the settings are used, the views must be
   * filtered for accessibility by the user.</p>
   *
   * <p>the {@link StructureAuth#getUser() current user} must have {@link PermissionLevel#ADMIN}
   * access level to the view being deleted.</p>
   *
   * @param viewId the ID of the view being removed
   * @throws StructureException if deletion could not succeed
   */
  void deleteView(@Nullable Long viewId) throws StructureException;

  /**
   * <p>Retrieves view settings for the specified structure. View settings define the associated views,
   * which are offered to the users in the "Views" drop-down.</p>
   *
   * <p>Unless {@link #setViewSettings} was previously called for a structure, the structure has
   * default view settings. Use {@link ViewSettings#isDefined()} to check if view settings
   * for the view has been adjusted from default.</p>
   *
   * <p>Unless security checks are {@link StructureAuth#isSecurityOverridden() disabled},
   * the method checks that the {@link StructureAuth#getUser() current user} has access
   * to the specified structure. Note that everyone has access to the global
   * {@link #getDefaultViewSettings() default view settings}.</p>
   *
   * <p>Note that retrieved {@code ViewSettings} may contain views that are not available for the user.
   * Before serving the views, make sure the user has at least {@code VIEW} access to them. See also
   * method {@link #getMenuItems}, which checks each view for accessibility.</p>
   *
   * @param structureId the ID of the structure
   * @return view settings
   * @throws StructureException if the user does not have {@code VIEW} access to the structure or structure does not exist
   */
  @NotNull
  ViewSettings getViewSettings(@Nullable Long structureId) throws StructureException;

  /**
   * <p>Updates view settings for the specified structure.</p>
   *
   * <p>Unless security checks are {@link StructureAuth#isSecurityOverridden() disabled},
   * the {@link StructureAuth#getUser() current user} must have {@code ADMIN} access level for the structure.</p>
   *
   * <p>If {@code settings} parameter is {@code null}, the settings for the structure are "cleared" and
   * will inherit the global default settings.</p>
   *
   * <p>Note that {@code ViewSettings} may contain views that are not available for a user, or even deleted views.
   * Before serving the views, make sure the user has at least {@code VIEW} access to them.</p>
   *
   * @param structureId the ID of the structure
   * @param settings the settings or {@code null} to reset settings to default
   * @throws StructureException if the user does not have the required permissions for this change
   */
  void setViewSettings(@Nullable Long structureId, @Nullable ViewSettings settings) throws StructureException;

  /**
   * <p>Retrieves the global default view settings, which apply to all structure that don't have
   * view settings overridden.</p>
   *
   * <p>Everyone has read access to the global default view settings.</p>
   *
   * @return global default view settings
   */
  @NotNull
  ViewSettings getDefaultViewSettings();

  /**
   * <p>Updates the global default view settings, which apply to all structure that don't have
   * view settings overridden.</p>
   *
   * <p>Only JIRA administrators may make this change.</p>
   *
   * <p>If {@code null} is passed as {@code settings}, the default view settings will be cleared, that is, they won't contain
   * any associated views.</p>
   *
   * @param settings the new global view settings
   * @throws StructureException if the user does not have the required permissions for this change
   */
  void setDefaultViewSettings(@Nullable ViewSettings settings) throws StructureException;

  /**
   * <p>This method retrieves a list of entries for the "Views" drop-down menu on the specified page, for
   * the specified structure. A menu item is basically a pair of {@code StructureView} and {@code boolean},
   * with the latter signifying that the view is the default.</p>
   *
   * <p>Note that due to the way default views are defined, there might be several default views in this
   * "menu". The calling method typically chooses the first entry that has true
   * {@link StructureViewMenuItem#isPreferred()}.</p>
   *
   * <p>The {@link StructureAuth#getUser() current user} must have at least {@code VIEW} access
   * to the specified structure, or the method will fail.</p>
   *
   * <p>At the moment, the following pages are supported:</p>
   * <ul>
   *   <li>{@link StructurePage#STRUCTURE_BOARD}</li>
   *   <li>{@link StructurePage#STRUCTURE_BOARD_WITH_DETAILS}</li>
   *   <li>{@link StructurePage#PROJECT_TAB}, {@link StructurePage#VERSION_TAB} and {@link StructurePage#COMPONENT_TAB} (have the same settings)</li>
   *   <li>{@link StructurePage#ISSUE_VIEW}</li>
   *   <li>{@link StructurePage#GADGET}</li>
   * </ul>
   *
   * <p>All other pages will default to {@link StructurePage#STRUCTURE_BOARD}.</p>
   *
   * @param structureId the ID of the structure, for which the menu list is retrieved
   * @param page page type for which the menu is being built
   * @return a list of menu items, may be empty
   */
  @NotNull
  List<StructureViewMenuItem> getMenuItems(@Nullable Long structureId, @Nullable StructurePage page);

  /**
   * <p>Retrieves all structures that are "associated" with the specified view, i.e. structures that
   * have overridden view settings that include the view.</p>
   *
   * <p>This method returns only structures that the {@link StructureAuth#getUser() current user}
   * can at least view.</p>
   *
   * <h3>Breaking change</h3>
   * <p>Prior to Structure API 10.0, this method also returned all structures that have default
   * view settings if the specified view was used in the default view settings. This is no longer
   * the case, for performance reasons. If you absolutely need to get the old behaviour,
   * you can simulate that by retrieving all structures that the user can view from {@link StructureManager}
   * and checking if their view settings are defined.</p>
   *
   * @param viewId the ID of the view
   * @return a list of structures that are associated with the specified view, perhaps empty
   * @throws StructureException if the user does not have access to the specified view
   */
  @NotNull
  List<Structure> getAssociatedStructures(@Nullable Long viewId) throws StructureException;
}
