package com.almworks.jira.structure.api.util;

import com.atlassian.fugue.Function2;
import com.atlassian.jira.util.json.JSONArray;
import com.atlassian.jira.util.json.JSONObject;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.*;

/**
 * A wrapper for a Map-based structure.
 *
 * It is a wrapper around JSONObject, which is a wrapper around Map. Compensating some of the JSONObject deficiencies. Made all methods optional.
 *
 * todo maybe get rid of using JSONObject - depends on the further development and usage
 *
 * Problems:
 * 1. If underlying map contains another Map as a value, JSONObject.getObject() won't return JSONObject.
 * 2. Awkward iteration of keys in JSONObject
 *
 */
public class MapObject {
  public static final MapObject EMPTY = new MapObject(Collections.<String, Object>emptyMap());

  public static final Function2<JSONArray, Integer, Long> LONG_ARRAY_EXTRACTOR = new Function2<JSONArray, Integer, Long>() {
    @Override
    public Long apply(JSONArray jsonArray, Integer index) {
      Object v = jsonArray.opt(index);
      return v instanceof Number ? ((Number) v).longValue() : null;
    }
  };
  public static final Function2<JSONArray, Integer, String> STRING_ARRAY_EXTRACTOR = new Function2<JSONArray, Integer, String>() {
    @Override
    public String apply(JSONArray jsonArray, Integer index) {
      return jsonArray.optString(index, null);
    }
  };
  public static final Function2<JSONArray, Integer, Object> RAW_OBJECT_ARRAY_EXTRACTOR = new Function2<JSONArray, Integer, Object>() {
    @Override
    public Object apply(JSONArray jsonArray, Integer index) {
      return jsonArray.opt(index);
    }
  };
  public static final Function2<JSONArray, Integer, MapObject> MAP_OBJECT_ARRAY_EXTRACTOR = new Function2<JSONArray, Integer, MapObject>() {
    @Override
    public MapObject apply(JSONArray array, Integer i) {
      return from(array.opt(i));
    }
  };

  @NotNull
  protected final JSONObject myObject;
  protected MapView myMapView;

  public MapObject(JSONObject object) {
    myObject = object == null ? new JSONObject() : object;
  }

  public MapObject(Map<String, Object> map) {
    myObject = new JSONObject(map);
  }

  public static MapObject from(Object obj) {
    if (obj instanceof JSONObject) return new MapObject((JSONObject) obj);
    if (obj instanceof Map) return new MapObject(StructureUtil.<String, Object>mapType().cast(obj));
    return null;
  }

  public static <T> List<T> transform(JSONArray array, Function2<JSONArray, Integer, T> extractor) {
    if (array == null || array.length() == 0) return Collections.emptyList();
    int length = array.length();
    ArrayList<T> r = new ArrayList<>(length);
    for (int i = 0; i < length; i++) {
      T value = extractor.apply(array, i);
      if (value != null) r.add(value);
    }
    return r;    
  }

  public boolean isEmpty() {
    return myObject.length() == 0;
  }

  public boolean has(String key) {
    return myObject.has(key);
  }
  /**
   * @param name
   * @return sub-object at key {@code} or empty MapObject
   */
  @Nullable
  public MapObject getObject(String name) {
    Object value = myObject.opt(name);
    if (value == null) return null;
    if (value instanceof JSONObject) {
      return new MapObject((JSONObject) value);
    }
    if (value instanceof Map) {
      return new MapObject(StructureUtil.<String, Object>mapType().cast(value));
    }
    return null;
  }

  @NotNull
  public MapObject traverse(String name) {
    MapObject r = getObject(name);
    return r == null ? EMPTY : r;
  }

  @NotNull
  public Iterable<String> keys() {
    return new Iterable<String>() {
      @Override
      public Iterator<String> iterator() {
        return myObject.keys();
      }
    };
  }

  @Nullable
  public Object get(String key) {
    return myObject.opt(key);
  }

  @Nullable
  public String getString(String key) {
    return myObject.optString(key, null);
  }

  public long getLong(String key) {
    return myObject.optLong(key, 0);
  }

  public int getInt(String key) {
    return myObject.optInt(key, 0);
  }

  public boolean getBoolean(String key) {
    return myObject.optBoolean(key);
  }

  @NotNull
  public List<Long> getLongList(@Nullable String key) {
    return getList(key, LONG_ARRAY_EXTRACTOR);
  }

  @NotNull
  public List<String> getStringList(@Nullable String key) {
    return getList(key, STRING_ARRAY_EXTRACTOR);
  }

  @NotNull
  public List<Object> getRawObjectList(@Nullable String key) {
    return getList(key, RAW_OBJECT_ARRAY_EXTRACTOR);
  }

  @NotNull
  public List<MapObject> getObjectList(@Nullable String key) {
    return getList(key, MAP_OBJECT_ARRAY_EXTRACTOR);
  }

  @NotNull
  public <T> List<T> getList(@Nullable String key, Function2<JSONArray, Integer, T> extractor) {
    if (key == null) return Collections.emptyList();

    // todo ListObject similar to MapObject
    JSONArray array;
    Object obj = myObject.opt(key);
    if (obj instanceof JSONArray) {
      array = (JSONArray) obj;
    } else if (obj instanceof Collection) {
      array = new JSONArray((Collection) obj);
    } else {
      array = null;
    }

    return transform(array, extractor);
  }

  public Map<String, Object> asImmutableMap() {
    if (myMapView == null) {
      myMapView = new MapView();
    }
    return myMapView;
  }

  @Override
  public String toString() {
    // todo account for inner MapObjects
    return myObject.toString();
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;
    // todo We can't use myObject.equals() because of the possibility of nested MapObjects
    MapObject mapObject = (MapObject) o;
    return myObject.equals(mapObject.myObject);
  }

  @Override
  public int hashCode() {
    return myObject.hashCode();
  }


  private class MapView extends AbstractMap<String, Object> {
    private SetView mySetView;

    @NotNull
    @Override
    public Set<Entry<String, Object>> entrySet() {
      if (mySetView == null) {
        mySetView = new SetView();
      }
      return mySetView;
    }

    @Override
    public Object get(Object key) {
      return myObject.opt((String) key);
    }

    @Override
    public boolean containsKey(Object key) {
      return myObject.has((String) key);
    }

    @Override
    public int size() {
      return myObject.length();
    }

    @Override
    public boolean isEmpty() {
      return myObject.length() == 0;
    }
  }


  private class SetView extends AbstractSet<Map.Entry<String, Object>> {
    @NotNull
    @Override
    public Iterator<Map.Entry<String, Object>> iterator() {
      return new SetIterator();
    }

    @Override
    public int size() {
      return myObject.length();
    }

    private class SetIterator implements Iterator<Map.Entry<String, Object>> {
      private final Iterator<String> keysIt = myObject.keys();

      @Override
      public boolean hasNext() {
        return keysIt.hasNext();
      }

      @Override
      public Map.Entry<String, Object> next() {
        String key = keysIt.next();
        Object value = myObject.opt(key);
        return new AbstractMap.SimpleImmutableEntry<>(key, value);
      }

      @Override
      public void remove() {
        throw new UnsupportedOperationException();
      }
    }
  }
}
