package com.almworks.jira.structure.api.structure.favorite;

import com.almworks.integers.LongArray;
import com.almworks.jira.structure.api.permissions.PermissionLevel;
import com.almworks.jira.structure.api.structure.Structure;
import com.atlassian.annotations.PublicApi;
import com.atlassian.jira.user.ApplicationUser;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.List;
import java.util.Map;

/**
 * <p>{@code StructureFavoriteManager} manages "favorite" relations between users and structures.</p>
 *
 * <p>Any user can mark a structure as their favorite, so they can quickly access it later.
 * Hence, a user has a set of favorite structures, and a structure has <i>popularity</i> -
 * the number of users who have marked it as their favorite.</p>
 *
 * <p>If a structure which is marked as favorite by a user becomes inaccessible for that user, it still
 * remains their favorite. If the user does not have the <tt>VIEW</tt> access to a structure, they shouldn't be
 * able to see it, whether it's their favorite or not.</p>
 *
 * <p>Except for {@link #getFavorites}, none of the methods of this class perform any permission checks,
 * so the calling code should check if the user has {@link PermissionLevel#VIEW} permission on their favorite
 * structures.</p>
 *
 * <p>Anonymous user does not have any favorites.</p>
 *
 * @since 7.2.0 (Structure 2.0)
 * @see com.almworks.jira.structure.api.structure.StructureManager
 */
@PublicApi
public interface StructureFavoriteManager {
  /**
   * Tells whether the structure is among user's favorite structures. This method does not check if the user has
   * {@link PermissionLevel#VIEW} on the structure and would return <code>true</code> for a favorite structure even if
   * the user no longer has the right to view it.
   * @param structureId the ID of the structure.
   * @param user the user.
   * @return <code>true</code> if the structure is favorite, <code>false</code> if otherwise or if any input parameter is
   * <code>null</code>.
   */
  boolean isFavorite(@Nullable Long structureId, @Nullable ApplicationUser user);

  /**
   * <p>Adds or removes the structure from user's favorites. This method does not check if the user has
   * {@link PermissionLevel#VIEW} on the structure and would perform even if the user doesn't have the right to view
   * the structure.</p>
   *
   * <p>Calling this method on the <tt>null</tt> user or structure has no effect.</p>
   *
   * @param structureId the ID of the structure.
   * @param user the user on whose favorites the operation should be performed.
   * @param favorite <code>true</code> if the structure should be made user's favorite, <code>false</code> if it
   *                 should be made not user's favorite.
   */
  void setFavorite(@Nullable Long structureId, @Nullable ApplicationUser user, boolean favorite);

  /**
   * Returns the number of users for whom the structure is favorite. If the result is shown to a user, the calling code
   * should ensure that the user has {@link PermissionLevel#VIEW} on the structure.
   *
   * @param structureId the ID of the structure.
   * @return the number of users for whom the structure is favorite. Returns 0 if structureId is <code>null</code>.
   */
  int getPopularity(@Nullable Long structureId);

  /**
   * <p>Returns a list of unarchived structures that are the user's favorite and that the user is allowed to see.</p>
   *
   * <p>The resulting list is sorted by structure name.</p>
   *
   * @param user the user whose favorites should be returned.
   * @return a list of the user's favorite unarchived structures. If user is <code>null</code>, an empty list is returned.
   */
  @NotNull
  List<Structure> getFavorites(@Nullable ApplicationUser user);

  /**
   * <p>Returns a list of structures that are the user's favorite and that the user is allowed to see.</p>
   *
   * <p>The resulting list is sorted by structure name.</p>
   *
   * @param user the user whose favorites should be returned.
   * @param includeArchivedStructures if true, result will also contain archived structures
   * @return a list of the user's favorite structures. If user is <code>null</code>, an empty list is returned.
   */
  @NotNull
  List<Structure> getFavorites(@Nullable ApplicationUser user, boolean includeArchivedStructures);

  @NotNull
  Map<String, LongArray> getAllFavorites();

  /**
   * <p>Sorts a list of structures by their popularity (descending). If the list is going to be shown to a user, the calling code
   * should ensure that the user has {@link PermissionLevel#VIEW} on every structure in the list.</p>
   *
   * <p>If two structures have the same popularity, they are sorted based on their name.</p>
   *
   * @param structures a modifiable list of structures for sorting.
   * @param user a user to whom the list will be shown. It is needed to properly sort structures with equal popularity
   *             (to use proper locale for name comparison). If <code>null</code>, the default system locale is used.
   *             No permission checks are made for this user.
   */
  void sortByPopularity(@Nullable List<Structure> structures, @Nullable ApplicationUser user);

  /**
   * <p>Filters a list of structures, producing another list with structures that match popularity criteria,
   * defined by minimum and maximum popularity.</p>
   *
   * <p>If the list is going to be shown to a user, the calling code
   * should ensure that the user has {@link PermissionLevel#VIEW} on every structure in the list.</p>
   *
   * <p>This method always returns writable list, even if the result is empty.</p>
   *
   * @param structures a list of structures for filtering, may be non-modifiable
   * @param minPopularity minimum popularity (inclusive). If you don't need to filter by minimum popularity, use {@code 0}.
   * @param maxPopularity maximum popularity (inclusive). If you don't need to filter by maximum popularity, use {@code Integer.MAX_VALUE}.
   * @return a new list with those structures that pass popularity filter, in the same order. The list is writable and
   * can be owned by the calling code, even if it is empty.
   */
  @NotNull
  List<Structure> filterByPopularity(@Nullable List<Structure> structures, int minPopularity, int maxPopularity);
}
