package com.almworks.jira.structure.api.row;

import com.almworks.integers.LongList;
import com.almworks.integers.wrappers.LongLongHppcOpenHashMap;
import com.almworks.jira.structure.api.forest.raw.Forest;
import com.atlassian.annotations.Internal;
import org.jetbrains.annotations.Nullable;

/**
 * {@code RowMapper} helps track the original row IDs for copied row. It is a temporary instance, received
 * from {@link RowManager#createMapper(Forest)}.
 *
 * <p>Mapping is done by traversing {@code originalId} links backwards, until {@code originalId} is zero and
 * while the row ID stay in the forest.</p>
 */
@Internal
public interface RowMapper {
  long mapRow(long requestedRow);

  Mapping map(LongList rows);


  class Mapping {
    private final LongList myRequestedRows;
    private final LongList myMappedRows;
    private final LongLongHppcOpenHashMap myMap;

    private Mapping(LongList requestedRows, LongList mappedRows, LongLongHppcOpenHashMap map) {
      myRequestedRows = requestedRows;
      myMappedRows = mappedRows;
      myMap = map;
    }

    public static Mapping identical(LongList rows) {
      return new Mapping(rows, rows, null);
    }


    public static Mapping mapped(LongList rows, LongList mappedRows, LongLongHppcOpenHashMap map) {
      return new Mapping(rows, mappedRows, map);
    }

    public LongList getRequestedRows() {
      return myRequestedRows;
    }

    /**
     * @return a mapped list of requested row. If a row was mapped, the mapped row ID is present instead of the requested ID.
     * IF row mapped to nothing, nothing is present. Otherwise, if row was not mapped (present in the forest), the requested
     * row ID is present.
     */
    public LongList getMappedRows() {
      return myMappedRows;
    }

    /**
     * @return a map with entries for the rows that are mapped to other rows -- rows that mapped to nothing (0) are
     * not present, as well as rows that were mapped to themselves
     *
     * Returns null when no mapping was made
     */
    @Nullable
    public LongLongHppcOpenHashMap getMap() {
      return myMap;
    }

    public boolean isMapped() {
      return myMap != null;
    }
  }
}
