package com.almworks.jira.structure.api.row;

import com.almworks.integers.*;
import com.almworks.jira.structure.api.forest.item.ItemForestBuilder;
import com.almworks.jira.structure.api.forest.raw.Forest;
import com.almworks.jira.structure.api.item.CoreIdentities;
import com.almworks.jira.structure.api.item.ItemIdentity;
import com.atlassian.annotations.Internal;
import com.atlassian.annotations.PublicApi;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.function.*;

/**
 * <p>{@code RowManager} is responsible for maintaining row database and row properties.</p>
 *
 * <p>Rows are a core concept in Structure architecture. A forest contains rows, not items. Rows are inserted and
 * moved around in the structure. However, rows are a technical concept -- the user deals with items. The main
 * responsibility of this component is to maintain mapping between rows and items they contain.</p>
 *
 * <p>Rows are identified by their {@code long} ID. Anywhere where you need to store rows, you need to store
 * its row ID -- you can later retrieve item ID from the {@code RowManager}.</p>
 *
 * <p>Rows are immutable. Once created, row information cannot be changed. Unused rows may be forgotten and later
 * reused or removed from the database by Row Manager.</p>
 *
 * <h3>Permanent and Transient Rows</h3>
 *
 * <p>Rows can be permanent and transient. Transient rows are used by the generators for the dynamic content.
 * As the content gets recalculated, new rows may be created, or the old ones may be reused.</p>
 *
 * <p>Transient rows live only in the JIRA instance's memory and not stored in the database. When JIRA restarts,
 * all transient rows are lost.</p>
 *
 * <p>On JIRA Data Center, transient row only lives on the node where it was created.
 * Another node will have its own version of that row, likely with a different row ID,
 * as all generators will work on that node separately.</p>
 *
 * <p>Transient rows can also be created not by generators but by manual action when they need to be inserted
 * into a transient forest source, such as a clipboard.</p>
 *
 * <p>Permanent rows are created for static structure content that is stored in the database. They are never
 * deleted and not reused. <em>In the coming version of Structure we will implement a "garbage collector" for
 * the unused permanent rows.</em></p>
 *
 * <h3>Temporary Rows</h3>
 *
 * <p>Note: there may be negative row IDs that are used for temporary rows when building forest fragments &mdash;
 * see {@link ItemForestBuilder}. Usage of these row ID must be confined to the builder they came from. Trying to
 * retrieve rows with negative ID from {@code RowManager} would result in {@link MissingRowException}.</p>
 *
 * @see StructureRow
 */
@PublicApi
public interface RowManager {
  /**
   * <p>Retrieves information about a structure row by its ID.</p>
   *
   * <p>Note that it is a runtime error to request a non-existing row, because that should never happen in a
   * correct code.</p>
   *
   * @param rowId row ID
   * @return row information
   * @throws MissingRowException if the specified row ID does not exist
   */
  @NotNull
  StructureRow getRow(long rowId) throws MissingRowException;

  /**
   * <p>Retrieves {@code StructureRow} given that the calling code has already checked the item for access by the user.
   * {@code itemVisible} parameter specifies whether the item is visible or not.</p>
   *
   * <p>This method is needed for optimization to avoid multiple access checks for an item.</p>
   *
   * @param rowId row ID
   * @param itemVisible if true, the corresponding item is known to be visible to the current user; if false, the item
   * is known to be invisible to the current user
   * @return row information. If the item is invisible, {@code StructureRow} is returned anyway, but {@link StructureRow#getItem}
   * will return {@code null}.
   * @throws MissingRowException if the specified row ID does not exist
   */
  @NotNull
  @Internal
  StructureRow getRow(long rowId, boolean itemVisible) throws MissingRowException;

  /**
   * Creates a new persistent row for the given item ID and semantics.
   *
   * @param itemId item ID
   * @param semantics semantics code. Semantics is still work in progress, and passing any value other
   * than 0 may result in undefined behavior.
   * @return row ID of the new row
   */
  long createRow(@NotNull ItemIdentity itemId, long semantics);

  /**
   * <p>Returns all rows created for the specified item.</p>
   *
   * <p><strong>Note:</strong> although this method returns transient rows as well, it cannot be relied upon
   * to find which dynamic structures contain a specific item, because a structure needs to be generated first.</p>
   *
   * <p>This method should be sufficiently fast, implementations should do indexing.</p>
   *
   * <p>This method will iterate through all the rows it finds before returning. If you want to stop scanning early,
   * please use {@link #findRows(ItemIdentity, LongPredicate)} instead.</p>
   *
   * @param itemId item ID
   * @return an iterator that provides row IDs of permanent and transient rows that contain this item
   */
  @NotNull
  default LongIterator findRows(@Nullable ItemIdentity itemId) {
    LongArray rows = new LongArray();
    findRows(itemId, rows::add);
    return rows.iterator();
  }

  /**
   * <p>Iterates through all rows created for the specified item.</p>
   *
   * <p><strong>Note:</strong> although this method returns transient rows as well, it cannot be relied upon
   * to find which dynamic structures contain a specific item, because a structure needs to be generated first.</p>
   *
   * <p>The {@code consumer} is called under a read lock, so it should be fast and take no locks.</p>
   *
   * <p>This method should be sufficiently fast, implementations should do indexing.</p>
   *
   * @param itemId item ID
   * @param consumer a consumer for found row IDs
   */
  default void findRows(@Nullable ItemIdentity itemId, @NotNull LongConsumer consumer) {
    findRows(itemId, i -> {
      consumer.accept(i);
      return true;
    });
  }

  /**
   * <p>Iterates through all rows created for the specified item.</p>
   *
   * <p><strong>Note:</strong> although this method returns transient rows as well, it cannot be relied upon
   * to find which dynamic structures contain a specific item, because a structure needs to be generated first.</p>
   *
   * <p>The {@code consumer} is called under a read lock, so it should be fast and take no locks.</p>
   *
   * <p>This method should be sufficiently fast, implementations should do indexing.</p>
   *
   * @param itemId item ID
   * @param consumer a consumer for found row IDs; must return {@code true} to continue iteration, {@code false} to stop
   */
  void findRows(@Nullable ItemIdentity itemId, @NotNull LongPredicate consumer);

  /**
   * <p>Bulk rows reading. Implementation should optimize this method for retrieving multiple rows at once.
   * Use it whenever you need to read potentially massive amount of rows, for example, when scanning the whole forest.</p>
   *
   * <p>The order in which iteratee is called is not guaranteed to be the same as rows. The same for missingCollector.</p>
   *
   * <p>Iteratee must be reasonably fast and avoid taking locks or accessing long-running services.</p>
   *
   * <p>It's possible to call other {@link RowManager} methods from inside the {@code iteratee}.</p>
   *
   * @param rows rows to read
   * @param sorted if true, then rows is sorted - can be used by the optimized code
   * @param missingCollector if not null, all missing rows will be added to the collector; if {@code null}, any missing
   * row will cause MissingRowException
   * @param iteratee predicate to call for each resolved row; if it returns false, the iteration stops
   * @throws MissingRowException if a row was not found and missingCollector is null
   */
  void scanRows(@Nullable LongIterator rows, boolean sorted, @Nullable LongCollector missingCollector,
    @NotNull Predicate<StructureRow> iteratee) throws MissingRowException;

  /**
   * <p>Convenience method that can be used to collect all issue IDs from given row IDs. Ignores missing rows.</p>
   *
   * @param rows a collection of rows
   * @param sorted if true, then rows is sorted - can be used by the optimized code
   * @param issuesCollector a collection to receive issue IDs
   * @param <C> type of the collector
   * @return the collector
   */
  @NotNull
  default <C extends LongCollector> C collectIssueIds(@Nullable LongIterable rows, boolean sorted,
    @NotNull C issuesCollector)
  {
    if (rows == null) return issuesCollector;
    scanRows(rows.iterator(), sorted, LongCollector.DUMMY, row -> {
      ItemIdentity itemId = row.getItemId();
      if (CoreIdentities.isIssue(itemId)) {
        issuesCollector.add(itemId.getLongId());
      }
      return true;
    });
    return issuesCollector;
  }

  /**
   * <p>Convenience method that can be used to collect all issue IDs from given row IDs. Ignores missing rows.</p>
   *
   * @param rows a collection of rows
   * @param issuesCollector a collection to receive issue IDs
   * @param <C> type of the collector
   * @return the collector
   */
  default <C extends LongCollector> C collectIssueIds(@Nullable LongIterable rows, C issuesCollector) {
    return collectIssueIds(rows, false, issuesCollector);
  }

  /**
   * <p>Creates a mapper. {@code RowMapper} is needed when, for one or more rows, you need to find the very original
   * row IDs.</p>
   *
   * @param forest forest that contains candidate rows and generators
   * @return a mapper
   * @see RowMapper
   */
  @Internal
  @NotNull
  RowMapper createMapper(@NotNull Forest forest);
}
