package com.almworks.jira.structure.api.permissions;

import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import javax.xml.bind.annotation.*;

/**
 * <p>This enumeration lists possible values for the access levels. Each user is assigned one of the
 * following levels, based on the permission rules set up for the structure.</p>
 *
 * <p>The permissions are linear - next permission level includes everything that was allowed
 * in the previous permission level, and a serial number defined for each level can be used in comparing levels.</p>
 *
 * <p>
 * Usually, when a permission level is passed as a parameter, it's supposed to be @Nullable. A
 * <code>null</code> value is treated as <code>NONE</code>.
 * </p>
 *
 * @author Igor Sereda
 * @see PermissionRule
 * @see <a href="http://wiki.almworks.com/display/structure/Structure+Permissions">Structure Permissions (Structure Documentation)</a>
 */
@XmlEnum
@XmlType(name = "level")
public enum PermissionLevel {
  // developer note: permissions must be declared in order of increasing serial number, otherwise fromSerial() will work incorrectly.

  /**
   * No permissions - the user is not allowed to view this structure or know that it exists.
   */
  @XmlEnumValue("0")NONE(0),

  /**
   * View permission - the user is allowed to view the structure, but they cannot change it.
   */
  @XmlEnumValue("10")VIEW(10),

  /**
   * The user is allowed to view and edit the structure.
   */
  @XmlEnumValue("20")EDIT(20),

  /**
   * The user is allowed to view and edit the structure, create and edit generators in it.
   */
  @XmlEnumValue("25")EDIT_GENERATORS(25),

  /**
   * The user is allowed to view and edit the structure, create and edit generators, and also change its properties and permissions and delete it.
   */
  @XmlEnumValue("30")ADMIN(30);

  private final int mySerial;

  private PermissionLevel(int serial) {
    mySerial = serial;
  }

  /**
   * @return serial number of the permission level
   */
  public int getSerial() {
    return mySerial;
  }

  /**
   * <p>
   * Used to check whether effective permission is enough to do something:
   * <code>if (permission.includes(PermissionLevel.EDIT))</code>
   * </p>
   * <p>
   * Any permission includes <code>NONE</code> level, and <code>null</code> permission level is treated as <code>NONE</code>.
   * </p>
   *
   * @param requiredPermission required permission level
   * @return true if this permission level includes everything that is allowed by requiredPermission level
   * @see StructureManager#isAccessible
   */
  public boolean includes(@Nullable PermissionLevel requiredPermission) {
    return requiredPermission == null || mySerial >= requiredPermission.getSerial();
  }

  /**
   * <p>Used to look up permission level by serial number. This is used in serialization or transferring
   * permissions.</p>
   *
   * <p>If serial number does not match one of the levels, the highest level with the serial number
   * less than specified will be used.</p>
   *
   * @param serial serial number
   * @return the corresponding permission level
   */
  @NotNull
  public static PermissionLevel fromSerial(int serial) {
    PermissionLevel r = NONE;
    for (PermissionLevel p : values()) {
      if (p.getSerial() > serial) break;
      r = p;
    }
    return r;
  }
}
