package com.almworks.jira.structure.api.forest.raw;

import com.almworks.integers.*;
import com.almworks.jira.structure.api.error.StructureErrors;
import com.almworks.jira.structure.api.error.StructureException;
import com.almworks.jira.structure.api.forest.ForestSource;
import com.almworks.jira.structure.api.forest.action.ForestAction;
import com.almworks.jira.structure.api.util.La;
import com.almworks.jira.structure.api.util.StructureUtil;
import com.atlassian.fugue.Pair;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.*;

/**
 * Mutable inheritor of {@link Forest} class.
 *
 * <code>ArrayForest</code> is a data structure that implements {@link Forest}.
 * Any modifications to the <code>ArrayForest</code> are only applied to that instance in memory.
 *
 * <h1>Under-After Coordinates</h1>
 * <p>
 * When a specific position in the forest needs to be defined for mutating operations such as move or insert, <code>under-after</code>
 * coordinates are used. (If operation addresses a specific row, the row ID is used instead.)
 * </p>
 * <p>
 * <code>Under-after</code> coordinates contain <code>under</code> value, which identifies the parent row of the
 * specified position (the depth and the subtree). If <code>under</code> is <code>0</code>, the position is at the
 * top (root row).
 * </p>
 * <p>
 * The other parameter, <code>after</code> coordinate, defines the location of the inserted row amongst siblings -
 * the rows at the same depth under the same parent. If <code>after</code> is <code>0</code>, it means that the position
 * should be the first under the specified parent.
 * </p>
 * <p>
 * Note: moving a row after itself, i.e. when the <code>under</code> coordinate is equal to the row's parent row
 * and the <code>after</code> coordinate is equal to the row itself, is allowed and does not change the position of the row.
 * </p>
 *
 * <p><strong>Methods from this class don't not modify the forest stored in the database for a structure.</strong>
 * To make modifications of the structure's forest, apply forest changes with {@link ForestSource#apply(ForestAction)}.
 * </p>
 *
 * <p>This class in <strong>not thread-safe</strong>.</p>
 *
 * @see ForestSource
 *
 * @author Igor Sereda
 */
public class ArrayForest implements Forest {
  private static final Logger logger = LoggerFactory.getLogger(ArrayForest.class);

  private WritableLongList myRows;
  private WritableIntList myDepths;
  private boolean myImmutable;

  public ArrayForest() {
    this(new LongArray(), new IntArray(), true);
  }

  /**
   * Creates a singular tree. Singular tree cannot be stored (the row is a standalone row), but can be used between
   * operations on other trees.
   *
   * @param row row ID of the root
   */
  public ArrayForest(long row) {
    this(LongArray.create(row), IntArray.create(0), true);
  }

  /**
   * Constructs a forest based on the given rows and depths.
   * <p/>
   * Row list and depth list must conform to the RowTree invariants. If invariants are violated, then either an
   * error is thrown (if assertions are on), or further behavior is undefined.
   *
   * @param rows list of row IDs
   * @param depths list of corresponding depths
   * @param reuseLists if true, passed instances of List can be used by this RowTree (otherwise a copy is made)
   */
  public ArrayForest(WritableLongList rows, WritableIntList depths, boolean reuseLists) {
    set(rows, depths, reuseLists);
  }

  public ArrayForest(LongList rows, IntList depths) {
    this(new LongArray(rows), new IntArray(depths), true);
  }

  public ArrayForest(Forest copyFrom) {
    this(copyFrom.getRows(), copyFrom.getDepths());
  }

  /**
   * <p>Replaces the contents of this forest with the values passed in the parameters.</p>
   *
   * <p>The passed lists must satisfy the invariant conditions listed in {@link Forest}, otherwise the results
   * are undefined.</p>
   *
   * @param rows new row list
   * @param depths new depths list
   * @param reuseLists if true, the passed arrays may be used by the forest without copying the data. In that case, the calling code must not use these lists after this method call.
   * @return this forest
   */
  public final Forest set(WritableLongList rows, WritableIntList depths, boolean reuseLists) {
    myRows = reuseLists ? rows : new LongArray(rows);
    myDepths = reuseLists ? depths : new IntArray(depths);
    assert checkInvariants();
    return this;
  }

  /**
   * <p>Replaces the contents of this forest with the values passed in the parameters.</p>
   *
   * <p>The passed lists must satisfy the invariant conditions listed in {@link Forest}, otherwise the results
   * are undefined.</p>
   *
   * @param rows new row list
   * @param depths new depths list
   * @return this forest
   */
  public Forest set(LongList rows, IntList depths) {
    return set(new LongArray(rows), new IntArray(depths), true);
  }

  /**
   * Checks whether RowTree invariants hold. Throws AssertionError if not, with a diagnostic message.
   *
   * @return true
   */
  boolean checkInvariants() {
    return checkInvariants(this);
  }

  static boolean checkInvariants(Forest f) {
    String problem = getDiagnostics(f);
    assert problem == null : problem;
    return true;
  }

  public String getDiagnostics() {
    return getDiagnostics(this);
  }

  /**
   * Checks whether RowTree invariants hold.
   *
   * @return null if all invariants are true, otherwise return a message with description of the problem
   */
  public static String getDiagnostics(Forest f) {
    LongList rows = f.getRows();
    IntList depths = f.getDepths();
    int size = rows.size();
    if (size != depths.size()) return "array size mismatch";
    if (size == 0) return null;
    if (depths.get(0) != 0) return "root not at 0 depth";

    Map<Long, Integer> rowMap = new HashMap<>();
    int depth = -1;
    for (int i = 0; i < size; i++) {
      long row = rows.get(i);
      Integer prev = rowMap.put(row, i);
      if (prev != null) {
        return "duplicate row @" + i + " " + row + " " + f;
      }
      int d = depths.get(i);
      if (d < 0) {
        return "bad depth @" + i + " " + d + " " + row + " " + f;
      }
      if (d > depth + 1) {
        return "bad depth change @" + i + " " + depth + " " + d + " " + row + " " + f;
      }
      depth = d;
    }
    return null;
  }

  public boolean containsRow(long row) {
    return myRows.contains(row);
  }

  /**
   * Convenience method to call {@link #mergeForest(Forest, long, long, ForestChangeEventHandler)} without event handler.
   *
   * @param forest the merged forest
   * @param under the parent of the merged forest, or 0 if the forest rows should be placed at the root level
   * @param after the preceding sibling of the merged forest, or 0 to place forest as the first child
   * @throws StructureException if the operation cannot be completed because it requires an invalid move
   */
  public void mergeForest(Forest forest, long under, long after) throws StructureException {
    mergeForest(forest, under, after, null);
  }

  public void addForest(long under, long after, Forest forest) {
    try {
      mergeForest(forest, under, after);
    } catch (StructureException e) {
      // todo get rid of StructureException? ForestException extends RuntimeException?
      // note that sometimes we need to know if add/merge was successful or with/without warnings - see SecuredForestSource
    }
  }

  public void clear() {
    assert checkInvariants();
    checkModification();
    myRows.clear();
    myDepths.clear();
  }


  /**
   * Invariant: after calling this method, either all rows from forest are in this forest, or exception is thrown.
   * <p/>
   * Merge is slow - more efficient algorithm can be employed if needed
   *
   * @param forest the source forest to merge into this forest
   * @param under under position
   * @param after after position
   * @param eventHandler
   * @throws StructureException if operation is not possible
   */
  /**
   * <p>Merges another forest into this forest. After the method has executed, this forest will contain all rows
   * from <code>forest</code> in the same topology under the specified positions: the root rows from <code>forest</code>
   * will be positioned as defined by <code>under-after</code> coordinates (see {@link Forest} for the explanation
   * of under-after coordinates), and non-root rows from <code>forest</code> will have the same parent rows as
   * in <code>forest</code>.</p>
   *
   * <p>This forest and the merged <code>forest</code> may have the same rows. In that case the rows are moved
   * within this forest and placed at the position required by this operation.</p>
   *
   * <p>
   * When <code>StructureException</code> is thrown from this method, the state of this forest is unknown - due to the
   * fact that merge splits into several atomic updates and the exception may be thrown after some of the updates
   * have taken place.
   * </p>
   *
   * @param forest the merged forest
   * @param under the parent of the merged forest, or 0 if the forest rows should be placed at the root level
   * @param after the preceding sibling of the merged forest, or 0 to place forest as the first child
   * @param eventHandler an optional event handler to get notifications about changes being applied - may be called several times
   * because merge could be split into series of moves and additions.
   * @throws StructureException if the operation cannot be completed because it requires an invalid move
   */
  public void mergeForest(
    Forest forest, long under, long after, @Nullable ForestChangeEventHandler eventHandler)
    throws StructureException
  {
    if (forest == null || forest.isEmpty()) return;
    assert checkInvariants();
    assert checkInvariants(forest);
    checkModification();

    if (isMutuallyExclusiveWith(forest)) {
      addForestMutuallyExclusive0(forest, under, after, eventHandler);
      return;
    }

    if (under != 0) {
      int parentIndex = myRows.indexOf(under);
      for (int p = parentIndex; p >= 0; p = getParentIndex(p)) {
        long pathRow = myRows.get(p);
        if (forest.containsRow(pathRow)) {
          throw new ForestMergeStructureException(under, pathRow, forest, this);
        }
      }
    }

    int size = forest.size();
    int i = 0;
    while (i < size) {
      long nextAfter = forest.getRows().get(i);
      i = mergeSubforest(forest, i, under, after, eventHandler);
      after = nextAfter;
    }

    assert checkInvariants();
  }

  /**
   * <p>Adds a single row at the specified position.</p>
   *
   * <p>If the row is already in the structure, this method moves it to the specified position.</p>
   *
   * @param row row to be added
   * @param under the parent of the row, or 0 if the row should be placed at the forest root level
   * @param after the preceding sibling of the row, or 0 to place row as the first child
   * @return true if the forest has been changed
   * @throws StructureException if <code>under</code> is not in the forest or if a similar problem happens
   */
  public boolean addRow(long row, long under, long after) throws StructureException {
    if (row == 0) return false;
    int idx = indexOf(row);
    if (idx >= 0) {
      moveSubtreeAtIndex(idx, under, after, row, null);
      return true;
    }
    addForestMutuallyExclusive0(new ArrayForest(row), under, after, null);
    return true;
  }

  private boolean isMutuallyExclusiveWith(Forest forest) {
    return StructureUtil.isMutuallyExclusive(this.getRows(), forest.getRows());
  }

  private int mergeSubforest(Forest forest, int index, long under, long after, ForestChangeEventHandler eventHandler)
    throws StructureException
  {
    LongList rows = forest.getRows();
    long row = rows.get(index);
    int thisIndex = this.indexOf(row); // don't confuse thisIndex with index
    if (thisIndex >= 0) {
      moveSubtreeAtIndex(thisIndex, under, after, row, eventHandler);
    } else {
      addForestMutuallyExclusive0(new ArrayForest(row), under, after, eventHandler);
    }
    IntList depths = forest.getDepths();
    int depth = depths.get(index);
    int size = forest.size();
    index++;
    long subAfter = 0;
    while (index < size && depths.get(index) > depth) {
      long nextSubAfter = rows.get(index);
      index = mergeSubforest(forest, index, row, subAfter, eventHandler);
      subAfter = nextSubAfter;
    }
    return index;
  }

  /**
   * Adds a forest to this forest. Works faster than {@link #addForest(long, long, Forest)} and
   * {@link #mergeForest(Forest, long, long)} when the added forest is guaranteed to be mutually exclusive
   * with this forest, i.e. when it contains only row IDs not present in this forest.
   * If this is not the case, the result of the operation is undefined.
   *
   * @param forest the forest to add
   * @param under the parent of the added forest, or 0 if the forest rows should be placed at the root level
   * @param after the preceding sibling of the added forest, or 0 to place forest as the first child
   * @throws StructureException if <code>under</code> is not in the forest or if a similar problem happens
   */
  public void addForestMutuallyExclusive(Forest forest, long under, long after) throws StructureException {
    addForestMutuallyExclusive0(forest, under, after, null);
  }

  private int addForestMutuallyExclusive0(Forest forest, long under, long after,
    ForestChangeEventHandler eventHandler) throws StructureException
  {
    if (forest == null || forest.isEmpty()) return -1;
    assert checkInvariants();
    assert checkInvariants(forest);
    checkModification();

    int parentIndex = getUnderIndex(under);
    int parentDepth = parentIndex < 0 ? -1 : myDepths.get(parentIndex);

    int insertAtIndex = parentIndex + 1;
    int insertAtDepth = parentDepth + 1;
    if (after != 0) {
      int siblingIndex = myRows.indexOf(after);
      if (siblingIndex < 0) {
        logger.info(this + ": ignoring invalid after: " + getDebugRowString(after));
      } else {
        siblingIndex = getPathIndexAtDepth(siblingIndex, insertAtDepth);
        if (siblingIndex < 0) {
          logger.info(this + ": ignoring invalid after: " + getDebugRowString(after) + " not at the right level");
        } else if (parentIndex != getParentIndex(siblingIndex)) {
          logger.info(this + ": ignoring invalid after: " + getDebugRowString(after) + " not under parent " + getDebugRowString(under));
        } else {
          insertAtIndex = getSubtreeEnd(siblingIndex);
        }
      }
    }

    LongList rows = forest.getRows();
    IntList depths = forest.getDepths();
    myRows.insertAll(insertAtIndex, rows);
    myDepths.insertAll(insertAtIndex, depths);
    if (insertAtDepth != 0) {
      for (int i = insertAtIndex + rows.size() - 1; i >= insertAtIndex; i--) {
        myDepths.set(i, myDepths.get(i) + insertAtDepth);
      }
    }
    if (eventHandler != null) {
      eventHandler.afterForestInserted(this, insertAtIndex, insertAtIndex + rows.size(), forest);
    }
    assert checkInvariants();
    return insertAtIndex;
  }

  private String getDebugRowString(long row) {
    // todo
    return String.valueOf(row);
  }

  private int getUnderIndex(long under) throws StructureException {
    int parentIndex;
    if (under == 0) {
      parentIndex = -1;
    } else {
      parentIndex = myRows.indexOf(under);
      if (parentIndex < 0) {
        throw StructureErrors.INVALID_FOREST_OPERATION.forRow(under).withMessage("cannot find under " + under + " in " + this);
      }
    }
    return parentIndex;
  }

  /**
   * Removes a subtree from this forest.
   * <p/>
   * This method will create a new <tt>RowTree</tt> with <tt>row</tt> at its root and all sub-rows, properly
   * outdented.
   *
   * @param row row ID of the root of the subtree to be removed
   * @return null if this tree does not contain <tt>row</tt>, otherwise an RowTree with the <tt>row</tt> as the
   * root
   */
  @NotNull
  public Forest removeSubtree(long row, ForestChangeEventHandler eventHandler) {
    assert checkInvariants();
    int rowIndex = myRows.indexOf(row);
    if (rowIndex < 0) return new ArrayForest();
    return removeSubtreeAtIndex(rowIndex, eventHandler);
  }

  /**
   * <p>Removes a sub-tree with rooted at the specified index from this forest.</p>
   *
   * @param index the index of the root of the sub-tree to be removed
   * @param eventHandler optional handler of the forest events
   * @return forest with the removed rows, or empty forest if <code>index</code> is negative
   */
  @NotNull
  public Forest removeSubtreeAtIndex(int index, @Nullable ForestChangeEventHandler eventHandler) {
    assert checkInvariants();
    checkModification();
    int lastIndex = getSubtreeEnd(index);
    Forest r = copySubtree0(index, lastIndex, false);
    if (eventHandler != null) {
      eventHandler.beforeSubtreeRemoved(this, index, lastIndex, r);
    }
    myRows.removeRange(index, lastIndex);
    myDepths.removeRange(index, lastIndex);
    assert checkInvariants();
    return r;
  }

  @NotNull
  private ArrayForest copySubtree0(int from, int to, boolean allowForest) {
    LongArray newRows = new LongArray(myRows.subList(from, to));
    IntArray newDepths = new IntArray(myDepths.subList(from, to));
    // Decrement depths. its separate tree so depths should start from 0
    int rowDepth = newDepths.get(0);
    if (rowDepth > 0) {
      for (int i = 0; i < newDepths.size(); i++) {
        int d = newDepths.get(i) - rowDepth;
        if (d < 0 || d == 0 && i > 0 && !allowForest) {
          throw new IllegalStateException("bad depth " + d + " @ " + i);
        }
        newDepths.set(i, d);
      }
    }
    return new ArrayForest(newRows, newDepths, true);
  }

  @NotNull
  public Forest subtree(long row) {
    assert checkInvariants();
    return subtreeAtIndex(indexOf(row));
  }

  @NotNull
  public Forest subtreeAtIndex(int index) {
    assert checkInvariants();
    if (index < 0) return new ArrayForest();
    int lastIndex = getSubtreeEnd(index);
    return copySubtree0(index, lastIndex, false);
  }

  public int getSubtreeEnd(int index) {
    if (index < 0) return 0;
    int size = myDepths.size();
    if (index > size) return size;
    int d = myDepths.get(index);
    int r = index + 1;
    for (; r < size; r++) {
      if (myDepths.get(r) <= d) break;
    }
    return r;
  }

  @Nullable
  @Override
  public ArrayForest copySubforest(long row) {
    if (row == 0) return copy();
    return copySubforestAtIndex(indexOf(row));
  }

  @Nullable
  @Override
  public ArrayForest copySubforestAtIndex(int k) {
    if (k < 0) return null;
    int p = getSubtreeEnd(k);
    if (p == k + 1) return new ArrayForest();
    return copySubtree0(k + 1, p, true);
  }

  @NotNull
  public LongList getRows() {
    return myRows;
  }

  @NotNull
  public IntList getDepths() {
    return myDepths;
  }

  public long getRow(int index) {
    return myRows.get(index);
  }

  public int getDepth(int index) {
    return myDepths.get(index);
  }

  public long getParent(long row) {
    if (row == 0) return 0;
    int index = myRows.indexOf(row);
    if (index < 0) return 0;
    int pi = getParentIndex(index);
    return pi < 0 ? 0 : myRows.get(pi);
  }

  public int getParentIndex(int index) {
    if (index < 0) return -1;
    int depth = myDepths.get(index);
    if (depth == 0) return -1;
    for (int i = index - 1; i >= 0; i--) {
      if (myDepths.get(i) == depth - 1) return i;
    }
    assert false : index + " " + this;
    return -1;
  }

  public int getPathIndexAtDepth(int index, int depth) {
    if (index < 0) return -1;
    if (depth < 0) return -1;
    if (myDepths.get(index) < depth) return -1;
    for (int i = index; i >= 0; i--) {
      if (myDepths.get(i) == depth) return i;
    }
    assert false : index + " " + this;
    return -1;
  }

  public int getPrecedingSiblingIndex(int index) {
    if (index <= 0) return -1;
    int depth = myDepths.get(index);
    for (int i = index - 1; i >= 0; i--) {
      int d = myDepths.get(i);
      if (d < depth) return -1;
      if (d == depth) return i;
    }
    assert false : index + " " + this;
    return -1;
  }

  public long getPrecedingSiblingForIndex(int index) {
    index = getPrecedingSiblingIndex(index);
    return index >= 0 ? getRow(index) : 0;
  }

  public long getPrecedingSibling(long row) {
    if (row == 0) return 0;
    int index = myRows.indexOf(row);
    return getPrecedingSiblingForIndex(index);
  }

  @NotNull
  public LongArray getPrecedingSiblings(long row) {
    return getPrecedingSiblingsForIndex(indexOf(row));
  }

  @NotNull
  public LongArray getPrecedingSiblingsForIndex(int index) {
    LongArray result = new LongArray();
    if (index <= 0) return result;
    int depth = myDepths.get(index);
    for (int i = index - 1; i >= 0; i--) {
      int d = myDepths.get(i);
      if (d < depth) break;
      if (d == depth) result.add(myRows.get(i));
    }
    result.reverse();
    return result;
  }

  @Override
  public int getNextSiblingIndex(int index) {
    if (index < 0 || index >= size() - 1) return -1;
    int depth = myDepths.get(index);
    for (int i = index + 1; i < size(); i++) {
      int d = myDepths.get(i);
      if (d < depth) return -1;
      if (d == depth) return i;
    }
    return -1;
  }

  @Override
  public long getNextSiblingForIndex(int index) {
    index = getNextSiblingIndex(index);
    return index >= 0 ? getRow(index) : 0;
  }

  @Override
  public long getNextSibling(long row) {
    if (row == 0) return 0;
    int index = myRows.indexOf(row);
    return getNextSiblingForIndex(index);
  }

  @NotNull
  public LongArray getChildren(long row) {
    return row == 0 ? getRoots() : getChildrenAtIndex(indexOf(row));
  }

  @NotNull
  public LongArray getChildrenAtIndex(int index) {
    LongArray children = new LongArray();
    if (index < 0) return children;
    int depth = myDepths.get(index);
    for (int i = index + 1; i < myDepths.size(); i++) {
      int d = myDepths.get(i);
      if (d <= depth) break;
      if (d == depth + 1) children.add(myRows.get(i));
    }
    return children;
  }

  @NotNull
  @Override
  public IntIterator getChildrenIndicesIterator(final int index) {
    return new IntFindingIterator() {
      int i = index;
      final int forestSize = size();
      final int parentDepth = index < 0 ? -1 : myDepths.get(index);

      @Override
      protected boolean findNext() {
        while (++i < forestSize) {
          int relDepth = myDepths.get(i) - parentDepth;
          if (relDepth <= 0) {
            return false;
          }
          if (relDepth == 1) {
            myNext = i;
            return true;
          }
        }
        return false;
      }
    };
  }

  @NotNull
  public LongArray getRoots() {
    LongArray r = new LongArray();
    for (int i = 0, size = myDepths.size(); i < size; i++) {
      if (myDepths.get(i) == 0) r.add(myRows.get(i));
    }
    return r;
  }


  @NotNull
  public ArrayForest filter(La<Long, ?> filter) {
    if (filter == null) return this;
    int size = myRows.size();
    int firstFiltered = getFirstNonMatchingRow(filter);
    if (firstFiltered == size) return this;

    LongArray rows = new LongArray(size);
    IntArray depths = new IntArray(size);
    int i = 0;
    while (i < size) {
      i = buildFilteredSubtree(rows, depths, i, 0, firstFiltered, filter);
    }
    return new ArrayForest(rows, depths, true);
  }

  private int getFirstNonMatchingRow(La<Long, ?> filter) {
    int firstFiltered = 0;
    int size = myRows.size();
    for (; firstFiltered < size; firstFiltered++) {
      if (!filter.accepts(myRows.get(firstFiltered))) break;
    }
    return firstFiltered;
  }

  private int buildFilteredSubtree(WritableLongList rows, WritableIntList depths, int index, int targetDepth,
    int firstFiltered, La<Long, ?> filter)
  {
    int size = myRows.size();
    if (index >= size) return index;
    int rootDepth = myDepths.get(index);
    int delta = targetDepth - rootDepth;
    int i = index;
    while (i < size) {
      long row = myRows.get(i);
      int depth = myDepths.get(i);
      if (depth < rootDepth || (depth == rootDepth && i > index)) break;
      boolean accepted = i < firstFiltered || (i > firstFiltered && filter.accepts(row));
      i++;
      if (accepted) {
        rows.add(row);
        depths.add(depth + delta);
      } else {
        while (i < size) {
          long nextDepth = myDepths.get(i);
          if (nextDepth <= depth) break;
          assert nextDepth == depth + 1 : i + " " + depth + " " + nextDepth;
          i = buildFilteredSubtree(rows, depths, i, depth + delta, firstFiltered, filter);
        }
      }
    }
    return i;
  }

  @NotNull
  public ArrayForest filterSoft(La<Long, ?> filter) {
    if (filter == null) return this;
    int size = myRows.size();
    int lastFiltered = size - 1;
    for (; lastFiltered >= 0; lastFiltered--) {
      if (!filter.accepts(myRows.get(lastFiltered))) break;
    }
    if (lastFiltered < 0) return this;
    LongArray revRows = new LongArray();
    IntArray revDepth = new IntArray();
    int lastDepth = 0;
    if (lastFiltered < size - 1) {
      revRows.addAll(myRows.subList(lastFiltered + 1, size));
      revDepth.addAll(myDepths.subList(lastFiltered + 1, size));
      revRows.reverse();
      revDepth.reverse();
      lastDepth = revDepth.get(revDepth.size() - 1);
    }
    for (int i = lastFiltered; i >= 0; i--) {
      int depth = myDepths.get(i);
      long row = myRows.get(i);
      boolean passes = depth < lastDepth || (i < lastFiltered && filter.accepts(row));
      if (passes) {
        revRows.add(row);
        revDepth.add(depth);
        lastDepth = depth;
      }
    }
    assert lastDepth == 0;
    revRows.reverse();
    revDepth.reverse();
    return new ArrayForest(revRows, revDepth, true);
  }

  @NotNull
  @Override
  public ArrayForest filterHardest(La<Long, ?> filter) {
    if (filter == null) return this;
    int size = myRows.size();
    int firstFiltered = getFirstNonMatchingRow(filter);
    if (firstFiltered == size) return this;

    LongArray rows = new LongArray(size);
    IntArray depths = new IntArray(size);
    int i = 0;
    while (i < size) {
      long row = myRows.get(i);
      boolean accepted = i < firstFiltered || (i > firstFiltered && filter.accepts(row));
      if (accepted) {
        rows.add(row);
        depths.add(myDepths.get(i));
        i++;
      } else {
        i = getSubtreeEnd(i);
      }
    }

    return new ArrayForest(rows, depths);
  }

  /**
   * Makes this instance non-modifiable
   *
   * @return row tree with the same data (backed by different collections), which cannot be modified
   */
  @NotNull
  public ArrayForest makeImmutable() {
    myImmutable = true;
    return this;
  }

  @Override
  public boolean isImmutable() {
    return myImmutable;
  }

  @NotNull
  public static Forest ensureImmutability(@NotNull Forest forest) {
    return forest.isImmutable() ? forest : new ArrayForest(forest).makeImmutable();
  }


  private void checkModification() {
    if (myImmutable) throw new UnsupportedOperationException();
  }


  /**
   * Gets the number of rows in this tree.
   *
   * @return the number of rows, also the size of lists returned by {@link #getRows()} and {@link #getDepths()}
   */
  public int size() {
    return myRows.size();
  }

  @NotNull
  public ArrayForest copy() {
    return new ArrayForest(myRows, myDepths);
  }

  /**
   * Convenience method to call {@link #moveSubtree(long, long, long, ForestChangeEventHandler)} without event handler.
   *
   * @param row the root row of the sub-tree being moved
   * @param under the new parent of the sub-tree, or 0 if the sub-tree should be placed at the forest root level
   * @param after the preceding sibling of the new location for the sub-tree, or 0 to place sub-tree as the first child
   * @return true if the sub-tree has been moved, false if not (for example, if the row is not in the forest)
   * @throws StructureException if the move is not possible - for example, <code>under</code> is not in the forest or if you attempt to move a sub-tree under itself
   */
  public boolean moveSubtree(long row, long under, long after) throws StructureException {
    return moveSubtree(row, under, after, null);
  }

  /**
   * <p>Moves sub-tree rooted at the specified row to a position specified by <code>(under, after)</code> coordinates.</p>
   *
   * <p>This method modifies the forest by removing the sub-tree with the specified row as the root and adding it
   * at the position specified by <code>under-after</code> coordinates. See {@link Forest} for the explanation
   * of under-after coordinates.</p>
   *
   * @param row the root row of the sub-tree being moved
   * @param under the new parent of the sub-tree, or 0 if the sub-tree should be placed at the forest root level
   * @param after the preceding sibling of the new location for the sub-tree, or 0 to place sub-tree as the first child
   * @param eventHandler optional handler of the move events
   * @return true if the sub-tree has been moved, false if not (for example, if the row is not in the forest)
   * @throws StructureException if the move is not possible - for example, <code>under</code> is not in the forest or if you attempt to move a sub-tree under itself
   */
  public boolean moveSubtree(long row, long under, long after,
    @Nullable ForestChangeEventHandler eventHandler) throws StructureException
  {
    assert checkInvariants();
    int idx = indexOf(row);
    if (idx < 0) return false;
    moveSubtreeAtIndex(idx, under, after, row, eventHandler);
    return true;
  }

  /**
   * <p>Moves sub-tree rooted at the specified index to a position specified by <code>(under, after)</code> coordinates.</p>
   *
   * <p>This method modifies the forest by removing the sub-tree with the specified row as the root and adding it
   * at the position specified by <code>under-after</code> coordinates. See {@link Forest} for the explanation
   * of under-after coordinates.</p>
   *
   * @param index the index of the root row of the sub-tree being moved
   * @param under the new parent of the sub-tree, or 0 if the sub-tree should be placed at the forest root level
   * @param after the preceding sibling of the new location for the sub-tree, or 0 to place sub-tree as the first child
   * @param eventHandler optional handler of the move events
   * @return new index of the sub-tree if it has been moved, -1 if not (for example, if the given row index is negative or the subtree is already there)
   * @throws StructureException if the move is not possible - for example, <code>under</code> is not in the forest or if you attempt to move a sub-tree under itself
   */
  public int moveSubtreeAtIndex(int index, long under, long after,
    @Nullable ForestChangeEventHandler eventHandler) throws StructureException
  {
    assert checkInvariants();
    if (index < 0) return -1;
    return moveSubtreeAtIndex(index, under, after, myRows.get(index), eventHandler);
  }

  /**
   * <p>Moves sub-tree rooted at the specified index to a position specified by <code>(under, after)</code> coordinates.</p>
   *
   * <p>This method modifies the forest by removing the sub-tree with the specified row as the root and adding it
   * at the position specified by <code>under-after</code> coordinates. See {@link Forest} for the explanation
   * of under-after coordinates.</p>
   *
   * @param index the index of the root row of the sub-tree being moved
   * @param under the new parent of the sub-tree, or 0 if the sub-tree should be placed at the forest root level
   * @param after the preceding sibling of the new location for the sub-tree, or 0 to place sub-tree as the first child
   * @param eventHandler optional handler of the move events
   * @return new index of the sub-tree if it has been moved, -1 if not (for example, if the given row index is negative)
   * @throws StructureException if the move is not possible - for example, <code>under</code> is not in the forest or if you attempt to move a sub-tree under itself
   */
  private int moveSubtreeAtIndex(int index, long under, long after, long row,
    ForestChangeEventHandler eventHandler) throws StructureException
  {
    assert checkInvariants();
    checkModification();
    if (!needsMove(row, index, under, after)) {
      return -1;
    }
    int parentIndex = getUnderIndex(under);
    if (parentIndex >= 0) {
      int underParentIndex = getPathIndexAtDepth(parentIndex, myDepths.get(index));
      if (underParentIndex == index) {
        throw StructureErrors.INVALID_FOREST_OPERATION.withMessage("moved row " + row + " is an ancestor of the destination parent row " + under);
      }
    }
    Forest forest = removeSubtreeAtIndex(index, eventHandler);
    int newIndex = addForestMutuallyExclusive0(forest, under, after, eventHandler);
    assert checkInvariants();
    assert newIndex >= 0;
    return newIndex;
  }

  private boolean needsMove(long row, int index, long under, long after) {
    int currUnderIndex = getParentIndex(index);
    long currUnder = currUnderIndex < 0 ? 0 : myRows.get(currUnderIndex);
    if (currUnder != under) {
      return true;
    }
    if (after == row) {
      return false;
    }
    int currAfterIndex = getPrecedingSiblingIndex(index);
    long currAfter = currAfterIndex < 0 ? 0 : myRows.get(currAfterIndex);
    return currAfter != after;
  }

  public int indexOf(long row) {
    return row == 0 ? -1 : myRows.indexOf(row);
  }

  public boolean isEmpty() {
    return myRows.isEmpty();
  }

  public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;

    Forest forest = (Forest) o;

    if (!myDepths.equals(forest.getDepths())) return false;
    if (!myRows.equals(forest.getRows())) return false;

    return true;
  }

  public int hashCode() {
    int result = myRows.hashCode();
    result = 31 * result + myDepths.hashCode();
    return result;
  }

  public String toString() {
    return toStringLimited(20);
  }

  @NotNull
  public String toFullString() {
    return toStringLimited(Integer.MAX_VALUE);
  }

  private String toStringLimited(int maxElements) {
    StringBuilder r = new StringBuilder("forest(");
    String prefix = "";
    if (myImmutable) {
      r.append("ro");
      prefix = ",";
    }
    int size = myRows.size();
    int len = Math.min(maxElements, size);
    for (int i = 0; i < len; i++) {
      r.append(prefix).append(myRows.get(i)).append(':').append(myDepths.get(i));
      prefix = ",";
    }
    if (len < size) r.append(" ... [").append(size).append(']');
    r.append(')');
    return r.toString();
  }

  @SuppressWarnings({"CloneDoesntDeclareCloneNotSupportedException"})
  public ArrayForest clone() {
    try {
      ArrayForest copy = (ArrayForest) super.clone();
      copy.myRows = new LongArray(copy.myRows);
      copy.myDepths = new IntArray(copy.myDepths);
      return copy;
    } catch (CloneNotSupportedException e) {
      throw new AssertionError(e);
    }
  }

  public long getLastChild(long parent) {
    int parentIndex;
    if (parent == 0) {
      parentIndex = -1;
    } else {
      parentIndex = indexOf(parent);
      if (parentIndex < 0) return 0;
    }
    return getLastChildByIndex(parentIndex);
  }

  public long getLastChildByIndex(int parentIndex) {
    int size = myDepths.size();
    if (parentIndex >= size) return 0;
    int depth = parentIndex < 0 ? 0 : myDepths.get(parentIndex) + 1;
    int index = -1;
    for (int i = parentIndex < 0 ? 0 : parentIndex + 1; i < size; i++) {
      int d = myDepths.get(i);
      if (d < depth) break;
      if (d == depth) {
        index = i;
      }
    }
    return index < 0 ? 0 : myRows.get(index);
  }

  public <T, C> C foldUpwards(ForestParentChildrenClosure<T, C> closure) {
    if (isEmpty()) return null;
    FoldControl fold = new FoldControl();
    C carry = null;
    while (fold.getIndex() > 0) {
      T result = visitUpwards0(fold, 0, closure);
      if (fold.isCancelled()) {
        return null;
      }
      carry = closure.combine(fold, result, carry);
    }
    return carry;
  }

  public void scanDownwards(ForestScanner scanner) {
    if (isEmpty()) return;
    ScanControl fold = new ScanControl();
    while (fold.getIndex() < size()) {
      scanner.acceptRow(fold, getRow(fold.getIndex()));
      if (fold.isCancelled()) {
        break;
      }
      fold.advance();
    }
  }

  /**
   * @param visitor the visitor to receive pairs of (parent, children)
   */
  public void visitParentChildrenUpwards(final ForestParentChildrenVisitor visitor) {
    foldUpwards(new ForestParentChildrenClosure<Void, Void>() {
      public Void combine(@NotNull ForestIterationControl fold, Void value, @Nullable Void carry) {
        return null;
      }

      public Void visitRow(@NotNull ForestIterationControl fold, long row, @NotNull LongList subrows, @Nullable Void carry) {
        boolean proceed = visitor.visit(ArrayForest.this, row, subrows);
        if (!proceed) fold.cancel();
        return null;
      }
    });
  }

  /**
   * Processes a subtree that ends at the specified index, and starts somewhere above it at the given depth.
   * Recursively processes nested subtrees first.
   *
   * @param fold fold control, index equals to {@link #getSubtreeEnd} of the subtree processed.
   * @param targetDepth the required depth of the root
   * @param closure closure
   * @return the result from the closure, in the fold the index equals to the root of the processed subtree
   */
  private <T, C> T visitUpwards0(FoldControl fold, int targetDepth, ForestParentChildrenClosure<T, C> closure) {
    int endIndex = fold.getIndex();
    if (endIndex <= 0) return null;

    int depth = myDepths.get(endIndex - 1);
    assert depth >= targetDepth : depth + " " + targetDepth;

    // does previous row contain a leaf of the required depth?
    if (depth == targetDepth) {
      return closure.visitRow(fold, fold.continueUp(depth), LongList.EMPTY, null);
    }

    // not a leaf - there will be children, allocate array
    LongArray children = fold.getChildrenContainer(targetDepth);

    // sub-iteration - gather nested sub-trees
    C carry = null;
    while (fold.getIndex() > 0 && myDepths.get(fold.getIndex() - 1) > targetDepth) {
      T result = visitUpwards0(fold, targetDepth + 1, closure);
      if (fold.isCancelled()) return null;
      children.add(myRows.get(fold.getIndex()));
      assert fold.getDepth() == targetDepth + 1;
      carry = closure.combine(fold, result, carry);
    }
    // we should exit by hitting the parent at the specified targetDepth
    assert fold.getIndex() > 0 && myDepths.get(fold.getIndex() - 1) == targetDepth : fold.getIndex() + " " + this;

    long parent = fold.continueUp(targetDepth);
    children.reverse();

    return closure.visitRow(fold, parent, children, carry);
  }

  @NotNull
  public LongArray getPath(long row) {
    return getPathForIndex(indexOf(row));
  }

  @NotNull
  public LongArray getPathForIndex(int idx) {
    LongArray r = new LongArray();
    if (idx < 0) return r;
    r.add(myRows.get(idx));
    for (int i = getParentIndex(idx); i >= 0; i = getParentIndex(i)) {
      r.add(myRows.get(i));
    }
    r.reverse();
    return r;
  }

  @NotNull
  public LongArray getParentPathForIndex(int index) {
    return getPathForIndex(getParentIndex(index));
  }

  /**
   * Convenience method to call {@link #removeSubtree(long, ForestChangeEventHandler)} without event handler.
   *
   * @param row the root of the sub-tree to be removed
   * @return forest with the removed rows, or empty forest if <code>row</code> is not in this forest.
   */
  @NotNull
  public Forest removeSubtree(long row) {
    return removeSubtree(row, null);
  }

  // todo add javadoc
  public void reorder(long parent, LongList children) {
    assert checkInvariants();
    checkModification();

    int from = -1;
    if (parent != 0) {
      from = indexOf(parent);
      if (from < 0) return; // exception? // todo yes, exception please - SecuredForestSource
    }
    int to = from < 0 ? size() : getSubtreeEnd(from);
    from++;
    if (to == from) return;
    int p = from;
    Map<Long, Pair<Integer, Integer>> segments = new HashMap<>(children.size());
    while (p < to) {
      int q = getSubtreeEnd(p);
      segments.put(getRow(p), Pair.pair(p, q));
      p = q;
    }
    assert children.size() == segments.size();
    LongArray rows = new LongArray(to - from);
    IntArray depths = new IntArray(to - from);
    for (LongIterator ii : children) {
      long child = ii.value();
      Pair<Integer, Integer> seg = segments.get(child);
      if (seg == null) {
        // todo error ? runtime exception? StructureException? ignore? note usage in SecuredForestSource
        throw new RuntimeException("bad parameter " + children + ", contains " + child + " that is not under " + parent);
      }
      rows.addAll(myRows.subList(seg.left(), seg.right()));
      depths.addAll(myDepths.subList(seg.left(), seg.right()));
    }
    myRows.setAll(from, rows);
    myDepths.setAll(from, depths);
  }

  // todo why? There are #addForest
  public void append(Forest forest) {
    checkModification();
    if (!isMutuallyExclusiveWith(forest)) {
      throw new IllegalArgumentException(this + " is not mutually exclusive with " + forest);
    }
    myRows.addAll(forest.getRows());
    myDepths.addAll(forest.getDepths());
    assert checkInvariants();
  }

  @NotNull
  @Override
  public LongIntIterator iterator() {
    return LongIntIterators.pair(getRows(), getDepths());
  }

  /**
   * Removes everything from under given row and inserts forest as sub-forest of rowId.
   *
   * @param rowId parent rowId, or 0 if the whole forest should be replaced
   * @param forest the content to insert in place of the removed sub-forest under rowId
   */
  public void replaceSubtrees(long rowId, Forest forest) {
    replaceSubtrees(rowId, forest, false);
  }

  public void replaceSubtreesMutuallyExclusive(long rowId, Forest forest) {
    replaceSubtrees(rowId, forest, true);
  }

  private void replaceSubtrees(long rowId, Forest forest, boolean sureMutuallyExclusive) {
    assert checkInvariants();
    assert checkInvariants(forest);

    int startReplaced, endReplaced;
    if (rowId == 0) {
      startReplaced = 0;
      endReplaced = myRows.size();
    } else {
      startReplaced = indexOf(rowId);
      if (startReplaced < 0) {
        throw new IllegalArgumentException("row " + rowId + " is not found");
      }
      endReplaced = getSubtreeEnd(startReplaced);
      startReplaced++;
    }

    if (sureMutuallyExclusive) {
      replaceRowsMutuallyExclusive(startReplaced, endReplaced, forest);
    } else {
      replaceRowsWithReallocation(rowId, startReplaced, endReplaced, forest);
    }

    assert checkInvariants();
  }

  private void replaceRowsMutuallyExclusive(int startReplaced, int endReplaced, Forest forest) {
    int diff = (endReplaced - startReplaced) - forest.size();
    if (diff < 0) {
      myRows.expand(startReplaced, -diff);
      myDepths.expand(startReplaced, -diff);
    } else if (diff > 0){
      myRows.removeRange(startReplaced, startReplaced + diff);
      myDepths.removeRange(startReplaced, startReplaced + diff);
    }

    myRows.setAll(startReplaced, forest.getRows());
    int addDepth = startReplaced > 0 ? myDepths.get(startReplaced - 1) + 1 : 0;
    if (addDepth > 0) {
      for (IntIterator it : forest.getDepths()) {
        myDepths.set(startReplaced++, it.value() + addDepth);
      }
    } else {
      myDepths.setAll(startReplaced, forest.getDepths());
    }
  }

  private void replaceRowsWithReallocation(long rowId, int startReplaced, int endReplaced, Forest forest) {
    int forestSize = size();
    int capacity = forestSize - (endReplaced - startReplaced) + forest.size();
    LongArray rows = new LongArray(capacity);
    IntArray depths = new IntArray(capacity);

    LongList preceding = myRows.subList(0, startReplaced);
    LongList succeeding = myRows.subList(endReplaced, forestSize);

    boolean mutuallyExclusive = StructureUtil.isMutuallyExclusive(forest.getRows(), LongCollections.concatLists(preceding, succeeding));

    // first part
    rows.addAll(preceding);
    depths.addAll(myDepths.subList(0, startReplaced));

    if (mutuallyExclusive) {
      // replaced part - can insert immediately
      rows.addAll(forest.getRows());
      int addDepth = startReplaced > 0 ? myDepths.get(startReplaced - 1) + 1 : 0;
      if (addDepth > 0) {
        for (IntIterator ii : forest.getDepths()) {
          depths.add(ii.value() + addDepth);
        }
      } else {
        depths.addAll(forest.getDepths());
      }
    }

    // third part
    rows.addAll(succeeding);
    depths.addAll(myDepths.subList(endReplaced, forestSize));

    set(rows, depths, true);

    if (!mutuallyExclusive) {
      // add/merge
      try {
        this.mergeForest(forest, rowId, 0);
      } catch (StructureException e1) {
        // what?!
      }
    }
  }


  private abstract class AbstractIterationControl implements ForestIterationControl {
    protected int myIndex;
    protected int myDepth;
    private boolean myCancelled;

    public AbstractIterationControl(int initialIndex) {
      myIndex = initialIndex;
    }

    public String toString() {
      return "[" + getIndex() + ":" + getDepth() + "]";
    }

    public Forest getForest() {
      return ArrayForest.this;
    }

    public void cancel() {
      myCancelled = true;
    }

    public boolean isCancelled() {
      return myCancelled;
    }

    public int getIndex() {
      return myIndex;
    }

    public int getDepth() {
      return myDepth;
    }
  }


  private class FoldControl extends AbstractIterationControl {
    private final List<LongArray> myContainerStack = new ArrayList<>();

    public FoldControl() {
      super(size());
    }

    public LongArray getChildrenContainer(int targetDepth) {
      while (myContainerStack.size() <= targetDepth) myContainerStack.add(new LongArray());
      LongArray r = myContainerStack.get(targetDepth);
      r.clear();
      return r;
    }

    public long continueUp(int assertedDepth) {
      assert myIndex > 0;
      myIndex--;
      assert ArrayForest.this.getDepth(myIndex) == assertedDepth : myIndex + " " + assertedDepth + " " + ArrayForest.this.getDepth(myIndex);
      myDepth = assertedDepth;
      return getRow(myIndex);
    }
  }


  private class ScanControl extends AbstractIterationControl implements ForestScanControl {
    private final IntArray myParentIndexPath = new IntArray(8);
    private int mySkipIndex = -1; // if >= 0, next iteration will skip to that index; must always be a "subtree end" of one of parents!
    private int mySubtreeEnd = 0; // cached value, 0 means not calculated

    public ScanControl() {
      super(0);
      advanceDepthAndPath();
    }

    public void advance() {
      advanceIndex();
      advanceDepthAndPath();
      mySubtreeEnd = 0;
    }

    private void advanceDepthAndPath() {
      if (myIndex < size()) {
        myDepth = ArrayForest.this.getDepth(myIndex);
        int pathElementsToAdd = myDepth - myParentIndexPath.size() + 1;
        if (pathElementsToAdd > 0) {
          assert pathElementsToAdd == 1 : myDepth + " " + myIndex + " " + myParentIndexPath;
          myParentIndexPath.insertMultiple(myParentIndexPath.size(), 0, pathElementsToAdd);
        }
        myParentIndexPath.set(myDepth, myIndex);
      } else {
        myDepth = 0;
      }
    }

    private void advanceIndex() {
      if (mySkipIndex >= 0) {
        if (mySkipIndex > myIndex) {
          myIndex = mySkipIndex;
        } else {
          assert false : myIndex + " " + mySkipIndex;
          myIndex++;
        }
        mySkipIndex = -1;
      } else {
        myIndex++;
      }
    }

    public int subtreeEnd() {
      if (mySubtreeEnd == 0) {
        mySubtreeEnd = getSubtreeEnd(myIndex);
      }
      return mySubtreeEnd;
    }

    public long getParent() {
      int parentIndex = getParentIndex();
      return parentIndex < 0 ? 0 : getRow(parentIndex);
    }

    public int getParentIndex() {
      if (myDepth == 0) return -1;
      assert myDepth < myParentIndexPath.size() : myDepth + " " + myParentIndexPath;
      return myParentIndexPath.get(myDepth - 1);
    }

    public int skipSubtree() {
      mySkipIndex = subtreeEnd();
      return mySkipIndex;
    }

    public int skipParentSubtree(int depth) {
      if (depth < -1) throw new IllegalArgumentException("cannot skip to depth " + depth);
      if (depth > myDepth) throw new IllegalArgumentException("cannot skip to the same or deeper level " + depth + this);
      int size = myDepths.size();
      if (depth == -1) {
        mySkipIndex = size;
      } else {
        for (mySkipIndex = myIndex + 1; mySkipIndex < size; mySkipIndex++) {
          if (myDepths.get(mySkipIndex) <= depth) break;
        }
      }
      return mySkipIndex;
    }

    public LongList getSubtreeRows() {
      return getRows().subList(myIndex, subtreeEnd());
    }

    public IntList getParentPathIndexes() {
      return myParentIndexPath.subList(0, myDepth);
    }
  }

}
