package com.almworks.jira.structure.api.forest.item;

import com.almworks.integers.LongArray;
import com.almworks.integers.LongIterator;
import com.almworks.jira.structure.api.forest.raw.ArrayForest;
import com.almworks.jira.structure.api.forest.raw.Forest;
import com.almworks.jira.structure.api.row.StructureRow;
import com.almworks.jira.structure.api.row.MissingRowException;
import com.atlassian.annotations.PublicApi;
import com.google.common.collect.ImmutableMap;
import org.jetbrains.annotations.NotNull;

import javax.annotation.concurrent.Immutable;
import java.util.Collections;
import java.util.Map;

@PublicApi
@Immutable
public final class ImmutableItemForest implements ItemForest {
  public static final ItemForest EMPTY = new ImmutableItemForest(
    new ArrayForest().makeImmutable(), ImmutableMap.<Long, StructureRow>of());

  private final Forest myForest;
  private final Map<Long, StructureRow> myRows;

  private ImmutableItemForest(@NotNull Forest forest, @NotNull Map<Long, StructureRow> rows) {
    myForest = forest;
    myRows = rows;
  }

  public static ImmutableItemForest of(@NotNull Forest forest, @NotNull Map<Long, StructureRow> rows) {
    checkAllRowsPresent(forest, rows);
    forest = new ArrayForest(forest).makeImmutable();
    rows = ImmutableMap.copyOf(rows);
    return new ImmutableItemForest(forest, rows);
  }

  public static ImmutableItemForest of(@NotNull StructureRow row) {
    return new ImmutableItemForest(new ArrayForest(row.getRowId()), Collections.singletonMap(row.getRowId(), row));
  }

  private static void checkAllRowsPresent(@NotNull Forest forest, @NotNull Map<Long, ?> rows) throws IllegalArgumentException {
    LongArray missing = null;
    for (LongIterator it : forest.getRows()) {
      if (rows.get(it.value()) == null) {
        if (missing == null) {
          missing = new LongArray();
        }
        missing.add(it.value());
      }
    }
    if (missing != null) {
      throw new IllegalArgumentException("No data for " + missing);
    }
  }

  @NotNull
  public static ImmutableItemForest copy(@NotNull ItemForest copyFrom) {
    if (copyFrom instanceof ImmutableItemForest) {
      return (ImmutableItemForest) copyFrom;
    } else {
      ArrayForest forest = new ArrayForest(copyFrom.getForest()).makeImmutable();
      ImmutableMap.Builder<Long, StructureRow> builder = ImmutableMap.builder();
      for (LongIterator it : forest.getRows()) {
        builder.put(it.value(), copyFrom.getRow(it.value()));
      }
      return new ImmutableItemForest(forest, builder.build());
    }
  }

  @NotNull
  public static ImmutableItemForest copySubtree(@NotNull ItemForest copyFrom, long rootId) {
    Forest subtree = copyFrom.getForest().subtree(rootId);
    ImmutableMap.Builder<Long, StructureRow> builder = ImmutableMap.builder();
    for (LongIterator it : subtree.getRows()) {
      builder.put(it.value(), copyFrom.getRow(it.value()));
    }
    return new ImmutableItemForest(subtree, builder.build());
  }

  @NotNull
  @Override
  public Forest getForest() {
    return myForest;
  }

  @NotNull
  @Override
  public StructureRow getRow(long rowId) throws MissingRowException {
    StructureRow row = myRows.get(rowId);
    if (row == null) {
      throw new MissingRowException(rowId);
    }
    return row;
  }
}
