package com.almworks.jira.structure.api.forest;

import com.almworks.jira.structure.api.pull.VersionedDataUpdate;
import com.atlassian.annotations.PublicApi;
import org.jetbrains.annotations.NotNull;

import java.util.List;

/**
 * Represents an update from {@link ForestSource}.
 *
 * @see ForestSource
 */
@PublicApi
public abstract class VersionedForestUpdate extends VersionedDataUpdate {
  @NotNull
  private final VersionedForest myLatest;

  @NotNull
  private final ForestSourceHealthStatus myHealth;

  private VersionedForestUpdate(@NotNull VersionedForest latest, @NotNull ForestSourceHealthStatus health) {
    super(latest.getVersion());
    myLatest = latest;
    myHealth = health;
  }

  /**
   * Returns the full forest. This method is provided for all updates (not only to full updates) because
   * {@link VersionedForest} is immutable and the source can just share the link to the most recent forest it has.
   */
  @NotNull
  public VersionedForest getLatest() {
    return myLatest;
  }

  /**
   * Casts this update to {@link Incremental} type. You first need to make sure this <em>is</em> an incremental type.
   *
   * @return this
   * @throws ClassCastException if the update is not incremental
   */
  @NotNull
  public Incremental asIncremental() {
    return (Incremental) this;
  }

  @NotNull
  public ForestSourceHealthStatus getHealth() {
    return myHealth;
  }

  /**
   * Represents a full update. The caller is supposed to use {@link #getLatest()} as the most recent state.
   */
  public static final class Full extends VersionedForestUpdate {
    public Full(VersionedForest latest, ForestSourceHealthStatus health) {
      super(latest, health);
    }

    public Full(VersionedForest latest) {
      super(latest, ForestSourceHealthStatus.HEALTHY);
    }

    @Override
    public boolean isFull() {
      return true;
    }
  }


  /**
   * Represents an incremental update. The caller is supposed to use {@link #getUpdates()} and apply the changes
   * to their old forest. Can also work as an empty update if there are no changes.
   */
  public static final class Incremental extends VersionedForestUpdate {
    @NotNull
    private final List<ForestChange> myUpdates;

    public Incremental(VersionedForest latest, @NotNull List<ForestChange> updates, ForestSourceHealthStatus health) {
      super(latest, health);
      //noinspection ConstantConditions
      if (updates == null) throw new IllegalArgumentException("updates cannot be null");
      myUpdates = updates;
    }

    public Incremental(VersionedForest latest, @NotNull List<ForestChange> updates) {
      this(latest, updates, ForestSourceHealthStatus.HEALTHY);
    }

    @NotNull
    public List<ForestChange> getUpdates() {
      return myUpdates;
    }

    @Override
    public boolean isEmpty() {
      return myUpdates.isEmpty();
    }
  }
}
