package com.almworks.jira.structure.api.cache.access;

import com.almworks.integers.*;
import com.almworks.jira.structure.api.forest.item.ItemForest;
import com.atlassian.annotations.Internal;
import com.atlassian.jira.user.ApplicationUser;
import org.jetbrains.annotations.NotNull;

/**
 * Provides per-row, per-issue caching and functions to verify row/issue visibility to the user
 */
@Internal
public interface ForestAccessCache {
  void clear();

  void clear(ApplicationUser user);

  /**
   * Calculates/retrieves the set of row IDs (among the passed {@code rows}), which contain items not visible to
   * the specified user.
   *
   * @param rows Rows to check -- only inserted rows must be passed here. Temporary rows (from ItemForest) are not allowed,
   * because this method works with RowManager and caches.
   */
  @NotNull
  LongSet getInvisibleRows(LongSizedIterable rows, String userKey) throws UserLookupException;

  /**
   * Convenient method to provide binary compatibility with old versions (that had {@code LongList} as first parameter).
   * @see #getInvisibleRows(LongSizedIterable, String)
   */
  @NotNull
  @Deprecated
  default LongSet getInvisibleRows(LongList rows, String userKey) throws UserLookupException {
    return getInvisibleRows((LongSizedIterable) rows, userKey);
  }

  /**
   * @see #getInvisibleRows(LongSizedIterable, String)
   */
  @NotNull
  LongSet getInvisibleRows(LongSizedIterable rows, ApplicationUser user);

  /**
   * Convenient method to provide binary compatibility with old versions (that had {@code LongList} as first parameter).
   * @see #getInvisibleRows(LongSizedIterable, String)
   */
  @NotNull
  @Deprecated
  default LongSet getInvisibleRows(LongList rows, ApplicationUser user) {
    return getInvisibleRows((LongSizedIterable) rows, user);
  }
  /**
   * Collects invisible rows from a temporary forest. Rows do not have to exist, and rows cache is not used by this
   * method. (Issue cache *is* used, after rows get converted to items.)
   *
   * @param forest a forest to check
   * @param userKey the user
   * @param invisibleRowsCollector collector that will receive row IDs for the rows that contain an item not visible to the user
   * @throws UserLookupException
   */
  void collectInvisibleRows(ItemForest forest, String userKey, LongCollector invisibleRowsCollector) throws UserLookupException;

  void collectInvisibleRows(ItemForest forest, ApplicationUser user, LongCollector invisibleRowsCollector);

  /**
   * Collects issues that are not visible to the specified user. Uses per-user visibility cache to speed up processing.
   *
   * @param issues issue IDs (in any order)
   * @param user the user
   * @param overrideSecurity when true, searching with overrideSecurity is used
   * @param invisibleIssuesCollector a collector to receive invisible issue IDs
   */
  void collectInvisibleIssues(LongIterable issues, ApplicationUser user, boolean overrideSecurity,
    LongCollector invisibleIssuesCollector);

  class UserLookupException extends Exception {
  }
}
