package com.almworks.jira.structure.api.backup;

import org.jetbrains.annotations.NotNull;

import java.io.File;

/**
 * <p>Backup operation allows to set backup parameters and run the backup.</p>
 *
 * <p>Note that for security reasons, if the file name is entered by the user, you must
 * check that it is located under the JIRA home directory, or force the file to be under that
 * directory. This operation trusts the caller and doesn't do any checking.</p>
 *
 * <p>If the specified backup file already exists, it will be overwritten.</p>
 */
public interface BackupOperation {
  /**
   * <p>Sets the target file for the backup. If the filename does not have extension,
   * ".xml" or ".zip" will be appended automatically, depending on the <code>useZip</code> flag.</p>
   *
   * <p>The file must be an absolute path.</p>
   *
   * @param file backup file
   * @return this operation
   * @throws IllegalArgumentException if the file is not a valid path
   */
  @NotNull
  BackupOperation setFile(@NotNull File file);

  /**
   * Sets the <code>useZip</code> flag. When the flag is set, the target file will be a zipped
   * XML file, otherwise it will be a plain XML file.
   *
   * @param useZip true if the file content should be compressed
   * @return this operation
   */
  @NotNull
  BackupOperation setUseZip(boolean useZip);

  /**
   * Sets the <code>backupHistory</code> flag. When the flag is set, full structure history
   * will be backed up, otherwise the target file will only contain the current state
   * of the structures.
   *
   * @param backupHistory true if structure history should be backed up
   * @return this operation
   */
  @NotNull
  BackupOperation setBackupHistory(boolean backupHistory);

  /**
   * Performs the backup synchronously.
   *
   * @return this operation
   * @throws Exception if file cannot be written or any other problem happens
   */
  @NotNull
  BackupOperation backup() throws Exception;

  /**
   * @return the file name that will be used for backup - that was set with {@link #setFile} method, but probably
   * with added extension.
   *
   * @throws IllegalStateException if the file has not been set
   */
  @NotNull
  File getFinalFile();
}
