package com.almworks.jira.structure.api.attribute.loader;

import com.almworks.jira.structure.api.attribute.AttributeSpec;
import com.almworks.jira.structure.api.item.ItemIdentity;
import com.almworks.jira.structure.api.row.StructureRow;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collections;
import java.util.Set;

public abstract class DerivedAttributeLoader<T, D> extends AbstractForestIndependentAttributeLoader<T> {
  protected final AttributeSpec<? extends D> myDependency;

  protected DerivedAttributeLoader(AttributeSpec<T> spec, AttributeSpec<? extends D> dependency) {
    super(spec);
    myDependency = dependency;
  }

  public final Set<? extends AttributeSpec<?>> getAttributeDependencies() {
    return Collections.singleton(myDependency);
  }

  @Override
  @NotNull
  public final AttributeValue<T> loadValue(StructureRow row, @NotNull Context context) {
    AttributeValue<? extends D> dependentResult = context.getAttributeValue(myDependency);
    if (dependentResult.isError()) {
      AttributeSpec<T> spec = getAttributeSpec();
      return spec == null ? AttributeValue.error() : dependentResult.cast(spec);
    }
    if (!dependentResult.isDefined()) {
      AttributeSpec<T> spec = getAttributeSpec();
      return spec == null ? AttributeValue.undefined() : dependentResult.cast(spec);
    }
    D dependentValue = dependentResult.getValue();
    T value = getValue(dependentValue, context);
    AttributeValue<T> result = AttributeValue.derived(value, dependentResult);
    if (dependentValue != null) {
      ItemIdentity trail = getItemTrail(dependentValue);
      if (trail != null) {
        result = result.withTrail(trail);
      }
    }
    return result;
  }

  protected ItemIdentity getItemTrail(@NotNull D value) {
    return null;
  }

  protected abstract T getValue(@Nullable D value, Context context);

  public static <T> AttributeLoader<T> idLoader(AttributeSpec<T> spec, AttributeSpec<T> dependency) {
    if (spec.getFormat() != dependency.getFormat()) return null;
    return new DerivedAttributeLoader<T, T>(spec, dependency) {
      @Override
      protected T getValue(@Nullable T value, Context context) {
        return value;
      }
    };
  }
}
