package com.almworks.jira.structure.api.attribute.loader;

import com.almworks.integers.LongSet;
import com.almworks.jira.structure.api.attribute.AttributeSpec;
import com.almworks.jira.structure.api.forest.item.ItemForest;
import com.almworks.jira.structure.api.row.StructureRow;
import com.almworks.jira.structure.api.util.La;
import com.google.common.base.Predicate;
import com.google.common.collect.Collections2;
import org.jetbrains.annotations.NotNull;

import java.util.*;

public abstract class CompositeAttributeLoader<T, Loader extends AttributeLoader<T>, LoaderValue, Params, LoaderContext extends AttributeLoader.Context>
  extends AbstractAttributeLoader<T> 
{
  public static final La<AttributeLoader<?>, Boolean> UNIVERSAL_LOADER = new La<AttributeLoader<?>, Boolean>() {
    public Boolean la(AttributeLoader<?> argument) {
      return argument.isEveryItemTypeSupported();
    }
  };
  public static final Predicate<AttributeLoader<?>> UNIVERSAL_LOADER_PRED = UNIVERSAL_LOADER.toPred();
  public static final Predicate<AttributeLoader<?>> SPECIFIC_LOADER_PRED = UNIVERSAL_LOADER.not().toPred();

  private final List<Loader> myLoaders;
  private final Collection<Loader> mySpecificLoaders;
  private final Collection<Loader> myUniversalLoaders;
  private final AttributeCachingStrategy myCachingStrategy;

  private CompositeAttributeLoader(AttributeSpec<T> spec, @NotNull List<Loader> loaders) {
    super(spec);
    myLoaders = loaders;
    mySpecificLoaders = Collections2.filter(myLoaders, SPECIFIC_LOADER_PRED);
    myUniversalLoaders = Collections2.filter(myLoaders, UNIVERSAL_LOADER_PRED);

    Loader shouldLoader = null, mustNotLoader = null;
    for (Loader loader : loaders) {
      if (loader.getCachingStrategy() == AttributeCachingStrategy.SHOULD) {
        shouldLoader = loader;
      }
      if (loader.getCachingStrategy() == AttributeCachingStrategy.MUST_NOT) {
        mustNotLoader = loader;
        break;
      }
    }
    myCachingStrategy = mustNotLoader != null ?
      AttributeCachingStrategy.MUST_NOT :
      shouldLoader != null ? AttributeCachingStrategy.SHOULD : AttributeCachingStrategy.MAY;
  }
  
  public static <T> AttributeLoader<T> create(AttributeSpec<T> spec, List<AttributeLoader<T>> loaders) {
    if (loaders == null || loaders.isEmpty()) {
      throw new IllegalArgumentException("empty or null loaders");
    }
    ForestDependencyType depType = ForestDependencyType.getType(loaders.get(0));
    for (int i = 1; i < loaders.size(); i++) {
      if (!depType.is(loaders.get(i))) {
        throw new IllegalArgumentException("cannot have different forest dependency in the composite loader");
      }
    }
    switch (depType) {
    case INDEPENDENT: return new ForestIndependentLoader<>(spec, (List)loaders);
    case AGGREGATE: return new AggregateLoader<>(spec, (List) loaders);
    case PROPAGATE: return new PropagateLoader<>(spec, (List) loaders);
    }
    throw new IllegalArgumentException("unknown forest dependency type " + depType);
  }
  
  public boolean isEveryItemTypeSupported() {
    return !myUniversalLoaders.isEmpty();
  }

  public boolean isItemTypeSupported(String itemType) {
    if (isEveryItemTypeSupported()) {
      return true;
    }
    for (AttributeLoader<T> loader : mySpecificLoaders) {
      if (loader.isItemTypeSupported(itemType)) {
        return true;
      }
    }
    return false;
  }

  public AttributeCachingStrategy getCachingStrategy() {
    return myCachingStrategy;
  }

  @NotNull
  public Set<? extends AttributeSpec<?>> getAttributeDependencies() {
    Set<AttributeSpec<?>> specs = new HashSet<>();
    for (AttributeLoader<T> loader : myLoaders) {
      specs.addAll(loader.getAttributeDependencies());
    }
    return specs;
  }

  protected final AttributeValue<LoaderValue> loadValue(LoaderContext context, Params additionalParams) {
    String itemType = context.getRow().getItemId().getItemType();
    for (Loader loader : mySpecificLoaders) {
      if (loader.isItemTypeSupported(itemType)) {
        AttributeValue<LoaderValue> result = apply(loader, additionalParams, context);
        if (result != null && !result.isEmpty()) {
          return result;
        }
      }
    }

    for (Loader loader : myUniversalLoaders) {
      AttributeValue<LoaderValue> result = apply(loader, additionalParams, context);
      if (result != null && !result.isEmpty()) {
        return result;
      }
    }
    return AttributeValue.undefined();
  }
  
  protected abstract AttributeValue<LoaderValue> apply(Loader loader, Params additionalParams, LoaderContext context);

  public boolean hasBulkLoaders() {
    return myLoaders.stream().anyMatch(CompositeAttributeLoader::isBulkLoader);
  }

  public static boolean isBulkLoader(AttributeLoader<?> loader) {
    return loader instanceof CompositeAttributeLoader ?
      ((CompositeAttributeLoader) loader).hasBulkLoaders() : loader instanceof BulkAttributeLoader;
  }

  public void preload(@NotNull LongSet rowIds, @NotNull ItemForest forest, @NotNull AttributeContext context) {
    myLoaders.forEach(loader -> preload(loader, rowIds, forest, context));
  }

  public static void preload(AttributeLoader<?> loader, @NotNull LongSet rowIds, @NotNull ItemForest forest, @NotNull AttributeContext context) {
    if (loader instanceof BulkAttributeLoader) {
      ((BulkAttributeLoader) loader).preload(rowIds, forest, context);
    } else if (loader instanceof CompositeAttributeLoader) {
      ((CompositeAttributeLoader) loader).preload(rowIds, forest, context);
    }
  }

  private static class ForestIndependentLoader<T> 
    extends CompositeAttributeLoader<T, AttributeLoader.ForestIndependent<T>, T, Void, Context> 
    implements AttributeLoader.ForestIndependent<T> 
  {
    private ForestIndependentLoader(AttributeSpec<T> spec, @NotNull List<ForestIndependent<T>> loaders) {
      super(spec, loaders);
    }

    @Override
    public AttributeValue<T> loadValue(StructureRow row, Context context) {
      return loadValue(context, null);
    }

    @Override
    protected AttributeValue<T> apply(ForestIndependent<T> loader, Void additionalParams, Context context) {
      return loader.loadValue(context.getRow(), context);
    }
  }

  private static class AggregateLoader<T> 
    extends CompositeAttributeLoader<T, AttributeLoader.Aggregate<T>, T, List<AttributeValue<T>>, AggregateContext<T>>
    implements AttributeLoader.Aggregate<T> 
  {
    private AggregateLoader(AttributeSpec<T> spec, @NotNull List<Aggregate<T>> loaders) {
      super(spec, loaders);
    }

    @Override
    public AttributeValue<T> loadValue(List<AttributeValue<T>> childrenValues, AggregateContext<T> context) {
      return loadValue(context, childrenValues);
    }

    @Override
    protected AttributeValue<T> apply(Aggregate<T> loader, List<AttributeValue<T>> childrenValues, 
      AggregateContext<T> context)
    {
      return loader.loadValue(childrenValues, context);
    }
  }

  private static class PropagateLoader<T>
    extends CompositeAttributeLoader<T, AttributeLoader.Propagate<T>, List<AttributeValue<T>>, List<StructureRow>, PropagateContext<T>> 
    implements AttributeLoader.Propagate<T> 
  {
    private PropagateLoader(AttributeSpec<T> spec, @NotNull List<Propagate<T>> loaders) {
      super(spec, loaders);
    }

    @Override
    public List<AttributeValue<T>> loadChildrenValues(AttributeValue<T> rowValue, List<StructureRow> children, 
      PropagateContext<T> context) 
    {
      return loadValue(context, children).getValue();
    }

    @Override
    protected AttributeValue<List<AttributeValue<T>>> apply(Propagate<T> loader, List<StructureRow> children, 
      PropagateContext<T> context)
    {
      List<AttributeValue<T>> childrenValues = loader.loadChildrenValues(context.getAttributeValue(), children, context);
      if (childrenValues != null) {
        for (AttributeValue<T> childrenValue : childrenValues) {
          if (childrenValue != null && childrenValue.isDefined()) {
            return AttributeValue.of(childrenValues);
          }
        }
      }
      return AttributeValue.undefined();
    }
  }
}
