package com.almworks.jira.structure.api.attribute.loader;

import com.almworks.jira.structure.api.attribute.AttributeSpec;
import com.almworks.jira.structure.api.attribute.TrailItemSet;
import com.almworks.jira.structure.api.item.ItemIdentity;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

/**
 * Represents a value, or lack thereof
 *
 * @param <T>
 */
public final class AttributeValue<T> {
  // note: there can be Undefined attribute values with trails and loader data
  private static final AttributeValue BARE_UNDEFINED = new AttributeValue<>(null, false, false, null, null);
  private static final AttributeValue BARE_ERROR = new AttributeValue<>(null, false, true, null, null);

  private final T myValue;
  private final boolean myDefined;
  private final boolean myError;
  private final TrailItemSet myAdditionalDataTrail;
  private final Object myLoaderData;

  private AttributeValue(T value, boolean defined, boolean error, TrailItemSet additionalDataTrail, Object loaderData) {
    myValue = value;
    myDefined = defined;
    myError = error;
    myAdditionalDataTrail = additionalDataTrail;
    myLoaderData = loaderData;
    // todo breaking API change -- maybe require that undefined and error values do not contain additional loader data or trail
  }

  @NotNull
  public static <T> AttributeValue<T> derived(T value, AttributeValue<?> dependentResult) {
    return new AttributeValue<>(value, value != null, false, dependentResult.getAdditionalDataTrail(), null);
  }

  @NotNull
  public static <T> AttributeValue<T> of(@Nullable T value) {
    return new AttributeValue<>(value, true, false, null, null);
  }

  @NotNull
  public static <T> AttributeValue<T> ofNullable(@Nullable T value) {
    return value == null ? undefined() : of(value);
  }

  @SuppressWarnings("unchecked")
  public static <T> AttributeValue<T> undefined() {
    return (AttributeValue<T>) BARE_UNDEFINED;
  }

  public AttributeValue<T> withTrail(ItemIdentity trailItem) {
    TrailItemSet trail = myAdditionalDataTrail;
    if (trail == null) {
      trail = TrailItemSet.None.NONE;
    }
    trail = trail.expand(trailItem);
    return new AttributeValue<>(myValue, myDefined, myError, trail, myLoaderData);
  }

  public AttributeValue<T> withTrail(TrailItemSet trail) {
    if (myAdditionalDataTrail != null) {
      throw new IllegalStateException("trail already set");
    }
    return new AttributeValue<>(myValue, myDefined, myError, trail, myLoaderData);
  }

  public AttributeValue<T> withData(Object loaderData) {
    return new AttributeValue<>(myValue, myDefined, myError, myAdditionalDataTrail, loaderData);
  }

  public T getValue() {
    return myValue;
  }

  public boolean isDefined() {
    return myDefined;
  }

  public boolean isError() {
    return myError;
  }

  public boolean isEmpty() {
    return !myDefined && myLoaderData == null;
  }

  @Nullable
  public <D> D getLoaderData(Class<D> valueClass) {
    return valueClass.isInstance(myLoaderData) ? valueClass.cast(myLoaderData) : null;
  }

  @NotNull
  public TrailItemSet getAdditionalDataTrail() {
    return myAdditionalDataTrail == null ? TrailItemSet.None.NONE : myAdditionalDataTrail;
  }

  public <X> AttributeValue<X> cast(AttributeSpec<X> spec) {
    if (myValue != null) {
      if (!spec.getFormat().getValueClass().isInstance(myValue)) {
        assert false : this + " " + spec;
        return null;
      }
    }
    //noinspection unchecked
    return (AttributeValue<X>) this;
  }

  @SuppressWarnings("unchecked")
  public static <T> AttributeValue<T> error() {
    return (AttributeValue<T>) BARE_ERROR;
  }

  public String toString() {
    if (myError) return "<!>";
    if (!myDefined) return "<?>";
    return String.valueOf(myValue);
  }
}
