package com.almworks.jira.structure.api.attribute;

import com.almworks.integers.LongList;
import com.almworks.jira.structure.api.attribute.loader.AttributeLoaderProvider;
import com.almworks.jira.structure.api.forest.ForestSpec;
import com.almworks.jira.structure.api.forest.item.ItemForest;
import com.almworks.jira.structure.api.forest.raw.Forest;
import com.almworks.jira.structure.api.row.RowManager;
import com.atlassian.annotations.Internal;
import com.atlassian.annotations.PublicApi;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;

/**
 * <p>{@code StructureAttributeService} provides unified way to retrieve field values, aggregate values or any other
 * defined attributes for issues or other items.</p>
 *
 * <p>An "attribute" is an abstraction that lets the consumer of this service not care about what type of value
 * is needed (is it just a field value or an aggregated value), how to calculate it, how to cache it and what type
 * of items is this value defined for.</p>
 *
 * <p>Furthermore, attribute system is extensible, so a third-party plugin can use attributes SPI to define their
 * own attribute or extend how existing attributes work for a new type of items. See {@link AttributeLoaderProvider}
 * for details.</p>
 *
 * <h3>Bulk Processing</h3>
 *
 * <p>This service is intended to be used for retrieving values for multiple attributes of multiple rows in one go.
 * Whenever you need to retrieve multiple values, try to minimize the number of calls to {@code StructureAttributeService}
 * for best performance.</p>
 *
 * <h3>Caching</h3>
 *
 * <p>The method that takes {@code ForestSpec} as the forest definition is preferred over the others because it
 * enables {@code StructureAttributeService} to cache the results. The forest specification serves as the cache key
 * and the service is able to correctly invalidate all cached values when the forest or items change.</p>
 *
 * <p>The methods that receive an arbitrary forest don't do caching and always recalculate the values.</p>
 *
 * <h3>Treating missing rows</h3>
 *
 * <p>Normally, if you request a value for a row ID that is not in the forest, you'll get an undefined value
 * in return. However, {@code StructureAttributeService} additionally checks if the row is a copy of another row,
 * made by a generator. In that case the calculation is performed for the original row. The resulting value can be
 * retrieved from the result by the row ID that was requested, so this conversion is transparent to the caller.</p>
 *
 * @see AttributeSpec
 * @see ForestSpec
 * @see VersionedRowValues
 * @see CoreAttributeSpecs
 */
@PublicApi
public interface StructureAttributeService {
  /**
   * <p>Returns attribute values for the given matrix of Rows and Attributes. The value is retrieved for each
   * {@code (row, attribute)} pair from the collections of rows and attributes passed as parameters.</p>
   *
   * <p>The values are cached when possible.</p>
   *
   * @param spec forest specification that identifies the forest, which contains the given rows
   * @param rows a collection of row IDs for which the values are needed
   * @param attributes a collection of attribute specifications for the values that are needed
   * @return an object that holds values for each pair of row ID and attribute, plus the versioning information
   * about the forest and items used in calculations
   */
  @NotNull
  VersionedRowValues getAttributeValues(@Nullable ForestSpec spec, @Nullable LongList rows,
    @Nullable Collection<? extends AttributeSpec<?>> attributes);

  /**
   * <p>Returns attribute values for the given matrix of Rows and Attributes. The value is retrieved for each
   * {@code (row, attribute)} pair from the collections of rows and attributes passed as parameters.</p>
   *
   * <p>The values are not cached because this method accepts an arbitrary forest. If you need to calculate
   * values for a structure or other forest that can be identified with {@link ForestSpec}, use
   * {@link #getAttributeValues(ForestSpec, LongList, Collection)}.</p>
   *
   * @param forest forest that contains the given rows
   * @param rows a collection of row IDs for which the values are needed
   * @param attributes a collection of attribute specifications for the values that are needed
   * @return an object that holds values for each pair of row ID and attribute, plus the versioning information
   * about the items used in calculations (it will have zero version for forest)
   */
  @NotNull
  VersionedRowValues getAttributeValues(@Nullable Forest forest, @Nullable LongList rows,
    @Nullable Collection<? extends AttributeSpec<?>> attributes);

  /**
   * <p>Returns attribute values for the given matrix of Rows and Attributes. The value is retrieved for each
   * {@code (row, attribute)} pair from the collections of rows and attributes passed as parameters.</p>
   *
   * <p>The values are not cached because this method accepts an arbitrary forest. If you need to calculate
   * values for a structure or other forest that can be identified with {@link ForestSpec}, use
   * {@link #getAttributeValues(ForestSpec, LongList, Collection)}.</p>
   *
   * <p>This method lets you calculate values for temporary rows, which are not yet known to {@link RowManager}. By
   * providing an instance of {@link ItemForest}, you have {@code StructureAttributeService} bypass going to row manager
   * for row data. Typically, negative row ID numbers are used for temporary rows.</p>
   *
   * @param forest forest that contains the given rows and information about each row
   * @param rows a collection of row IDs for which the values are needed
   * @param attributes a collection of attribute specifications for the values that are needed
   * @return an object that holds values for each pair of row ID and attribute, plus the versioning information
   * about the items used in calculations (it will have zero version for forest)
   */
  VersionedRowValues getAttributeValues(@Nullable ItemForest forest, @Nullable LongList rows,
    @Nullable Collection<? extends AttributeSpec<?>> attributes);

  /**
   * <p>This is the same method as {@link StructureAttributeService#getAttributeValues(ItemForest, LongList, Collection)},
   * except that base forest spec, from which the rows may have come, is provided. Used by generators to calculate
   * intermediate values.</p>
   */
  @Internal
  VersionedRowValues getAttributeValues(@Nullable ItemForest forest, @Nullable LongList rows,
    @Nullable Collection<? extends AttributeSpec<?>> attributes, @Nullable ForestSpec baseForestSpec);
}
