package com.almworks.jira.structure.api.attribute;

import com.almworks.jira.structure.api.util.JsonMapUtil;
import com.atlassian.annotations.PublicApi;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.*;

@PublicApi
public class AttributeSpecBuilder<T> {
  private String myId;
  private ValueFormat<T> myFormat;
  private ParamsBuilder<AttributeSpecBuilder<T>> myParams;

  public static AttributeSpecBuilder<Void> create() {
    return new AttributeSpecBuilder<>();
  }

  public static AttributeSpecBuilder<Void> create(String id) {
    return new AttributeSpecBuilder<Void>().setId(id);
  }

  public static <T> AttributeSpecBuilder<T> create(String id, ValueFormat<T> format) {
    return new AttributeSpecBuilder<Void>().setId(id).setFormat(format);
  }

  public static <T> AttributeSpecBuilder<T> create(String id, ValueFormat<T> format, Map<String, Object> params) {
    return new AttributeSpecBuilder<Void>()
      .setId(id)
      .setFormat(format)
      .params().copyFrom(params).done();
  }

  public AttributeSpec<T> build() {
    return new AttributeSpec<>(myId, myFormat, myParams.buildMap(), true);
  }

  public static <T> AttributeSpecBuilder<T> create(AttributeSpec<T> sample) {
    if (sample == null) return new AttributeSpecBuilder<>();
    return create(sample.getId(), sample.getFormat(), sample.getParamsMap());
  }

  public AttributeSpecBuilder<T> setId(String id) {
    myId = id;
    return this;
  }

  @SuppressWarnings("unchecked")
  public <R> AttributeSpecBuilder<R> setFormat(ValueFormat<R> format) {
    myFormat = (ValueFormat) format;
    return (AttributeSpecBuilder<R>) this;
  }

  public ParamsBuilder<AttributeSpecBuilder<T>> params() {
    if (myParams == null) {
      myParams = new ParamsBuilder<>(this);
    }
    return myParams;
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;

    AttributeSpecBuilder<?> that = (AttributeSpecBuilder<?>) o;

    if (myId != null ? !myId.equals(that.myId) : that.myId != null) return false;
    if (myFormat != null ? !myFormat.equals(that.myFormat) : that.myFormat != null) return false;
    return !(myParams != null ? !myParams.equals(that.myParams) : that.myParams != null);

  }

  @Override
  public int hashCode() {
    int result = myId != null ? myId.hashCode() : 0;
    result = 31 * result + (myFormat != null ? myFormat.hashCode() : 0);
    result = 31 * result + (myParams != null ? myParams.hashCode() : 0);
    return result;
  }

  @Override
  public String toString() {
    return "AttributeSpecBuilder('" + myId + "', " + myFormat + ", " + myParams + ")";
  }


  public class ParamsBuilder<P> {
    private final P myParent;
    private final Map<String, Object> myParams = new LinkedHashMap<>();

    public ParamsBuilder(P parent) {
      myParent = parent;
    }

    public P done() {
      return myParent;
    }

    public ParamsBuilder<ParamsBuilder<P>> object(String key) {
      Object existing = myParams.get(key);
      ParamsBuilder<ParamsBuilder<P>> r;
      if (existing instanceof ParamsBuilder) {
        //noinspection unchecked
        r = (ParamsBuilder<ParamsBuilder<P>>) existing;
      } else {
        r = new ParamsBuilder<>(this);
        put(key, r);
      }
      return r;
    }

    public ParamsBuilder<P> set(String key, Object value) {
      if (value == null) return this;
      if (value instanceof AttributeSpec) {
        return setAttribute(key, (AttributeSpec) value);
      }
      JsonMapUtil.checkValidParameter(value);
      return setValidated(key, value);
    }

    public ParamsBuilder<P> setAttribute(AttributeSpec<?> value) {
      return setAttribute(CoreAttributeSpecs.Param.ATTRIBUTE, value);
    }

    public ParamsBuilder<P> setAttribute(String key, AttributeSpec<?> value) {
      return object(key)
        .set(CoreAttributeSpecs.Param.ID, value.getId())
        .set(CoreAttributeSpecs.Param.FORMAT, value.getFormat().getFormatId())
        .setValidatedMap(CoreAttributeSpecs.Param.PARAMS, value.getParamsMap())
        .done();
    }

    public ParamsBuilder<P> copyFrom(@Nullable Map<String, Object> map) {
      if (map == null || map.isEmpty()) return this;
      JsonMapUtil.checkValidParameter(map);
      return copyFromValidated(map);
    }

    private ParamsBuilder<P> setValidated(String key, Object value) {
      if (value instanceof Map) {
        //noinspection unchecked
        return setValidatedMap(key, (Map) value);
      } else {
        put(key, value);
      }
      return this;
    }

    private ParamsBuilder<P> setValidatedMap(String key, Map<String, Object> map) {
      if (map == null || map.isEmpty()) return this;
      return object(key).copyFromValidated(map).done();
    }

    @NotNull
    private ParamsBuilder<P> copyFromValidated(Map<String, Object> map) {
      if (map != null) {
        for (Map.Entry<String, Object> e : map.entrySet()) {
          setValidated(e.getKey(), e.getValue());
        }
      }
      return this;
    }

    private void put(String key, Object r) {
      if (key == null) {
        throw new IllegalArgumentException("null key is not allowed");
      }
      myParams.put(key, r);
    }

    @Override
    public boolean equals(Object o) {
      if (this == o) return true;
      if (o == null || getClass() != o.getClass()) return false;
      ParamsBuilder<?> that = (ParamsBuilder<?>) o;
      return myParams.equals(that.myParams);
    }

    @Override
    public int hashCode() {
      return myParams.hashCode();
    }

    @Override
    public String toString() {
      return myParams.toString();
    }

    @Nullable
    public Map<String, Object> buildMap() {
      if (myParams.isEmpty()) return null;
      LinkedHashMap<String, Object> r = new LinkedHashMap<>(myParams);
      for (Iterator<Map.Entry<String, Object>> ii = r.entrySet().iterator(); ii.hasNext(); ) {
        Map.Entry<String, Object> e = ii.next();
        Object value = e.getValue();
        if (value instanceof ParamsBuilder) {
          value = ((ParamsBuilder<?>)value).buildMap();
          if (value == null) {
            ii.remove();
          } else {
            e.setValue(value);
          }
        }
      }
      return r;
    }

    public AttributeSpec<T> build() {
      return AttributeSpecBuilder.this.build();
    }
  }
}
