package com.almworks.jira.structure.api.attribute;

import com.almworks.jira.structure.api.attribute.loader.AttributeLoader;
import com.almworks.jira.structure.api.util.*;
import com.atlassian.annotations.PublicApi;
import org.apache.commons.lang.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import javax.annotation.concurrent.Immutable;
import java.util.Collections;
import java.util.Map;

/**
 * <p>{@code AttributeSpec} is the "attribute specification", a composite identifier of an attribute. See
 * {@link com.almworks.jira.structure.api.attribute package documentation} for the definition of an attribute.</p>
 *
 * <p>Attribute specification contains the following parts:</p>
 *
 * <ol>
 *   <li>A unique {@code String} identifier. Attributes that have the same identifier are considered to be semantically
 *   same or similar.</li>
 *
 *   <li>An unbounded parameters {@code Map}, which should be serializable to JSON (contain only simple types, maps
 *   and arrays). Two specifications with the same ID and parameters are considered to be
 *   semantically same.</li>
 *
 *   <li>{@link ValueFormat}, which defines the type and possible operations on the resulting value.</li>
 * </ol>
 *
 * <p>Use {@link AttributeSpecBuilder} to build an instance of {@code AttributeSpec} in the code.</p>
 *
 * @param <T> type of the value expected for this attribute
 * @see StructureAttributeService
 * @see AttributeSpecBuilder
 */
@PublicApi
@Immutable
public class AttributeSpec<T> {
  @NotNull
  private final String myId;

  @NotNull
  private final ValueFormat<T> myFormat;

  @NotNull
  private final Map<String, Object> myParams;

  private transient int myHashCode;
  private transient SpecParams myParamsObject;

  /**
   * Constructs an attribute spec with the given ID and format, without parameters.
   *
   * @param id attribute ID
   * @param format value format
   */
  public AttributeSpec(String id, ValueFormat<T> format) {
    this(id, format, null, false);
  }

  /**
   * Constructs an attribute spec with the given ID, format and parameters. The latter should contain only simple types
   * and collections, the things serializable into JSON.
   *
   * @param id attribute ID
   * @param format value format
   * @param params parameters map
   */
  public AttributeSpec(@NotNull String id, @NotNull ValueFormat<T> format, @Nullable Map<String, Object> params) {
    this(id, format, params, false);
  }

  AttributeSpec(@NotNull String id, @NotNull ValueFormat<T> format, @Nullable Map<String, Object> params, boolean reuseParams) {
    if (StringUtils.isBlank(id)) {
      throw new IllegalArgumentException("attribute id must not be empty");
    }
    if (id.length() > Limits.MAX_SPEC_ID_LENGTH) {
      throw new IllegalArgumentException("attribute id must not be longer than " + Limits.MAX_SPEC_ID_LENGTH + " chars");
    }
    if (id.indexOf(':') >= 0) {
      throw new IllegalArgumentException("attribute id must not contain colon");
    }
    if (format == null) {
      throw new IllegalArgumentException("attribute format must not be null");
    }
    myId = id;
    myFormat = format;
    Map<String, Object> map = reuseParams ? params : JsonMapUtil.copyParameters(params, true, true, true);
    // todo optimize - this should be done in copyParameters(), however, it is a more generic method, so it should have some abstraction
    myParams = AttributeSpecNormalization.normalizeParams(map != null ? map : Collections.emptyMap());
  }

  /**
   * Returns the attribute's ID.
   */
  @NotNull
  public String getId() {
    return myId;
  }

  /**
   * Returns the attribute's format.
   */
  @NotNull
  public ValueFormat<T> getFormat() {
    return myFormat;
  }

  /**
   * Returns the attribute's parameters as a read-only map. If there are no parameters, empty map is returned.
   */
  @NotNull
  public Map<String, Object> getParamsMap() {
    return myParams;
  }

  /**
   * Returns the same as {@link #getParamsMap()}, but wrapped into accessor object.
   */
  @NotNull
  public SpecParams getParams() {
    SpecParams paramsObject = myParamsObject;
    if (paramsObject == null) {
      myParamsObject = paramsObject = new SpecParams(myParams);
    }
    return paramsObject;
  }

  /**
   * Checks if this attribute specification is for the given attribute ID.
   *
   * @param id attribute's ID
   * @return true if this attribute has the same ID
   */
  public boolean is(String id) {
    return myId.equals(id);
  }

  /**
   * Checks if this attribute specification contains the given format.
   *
   * @param format value format
   * @return true if this attribute has this format
   */
  public boolean is(ValueFormat<?> format) {
    return myFormat.equals(format);
  }

  /**
   * Checks if this attribute specification is for the given ID and format.
   *
   * @param id attribute's ID
   * @param format value format
   * @return true if this attribute has this ID and format
   */
  public boolean is(String id, ValueFormat<?> format) {
    return is(id) && is(format);
  }

  /**
   * Returns an attribute spec with the same ID and parameters, but with the given {@code ValueFormat}.
   *
   * @param format the format for a new {@code AttributeSpec}
   */
  public <V> AttributeSpec<V> as(ValueFormat<V> format) {
    return is(format) ? format.cast(this) : new AttributeSpec<V>(myId, format, myParams);
  }

  /**
   * Returns a new attribute spec with added parameter.
   *
   * @param name parameter name
   * @param value parameter value
   * @return a new, adjusted attribute specification
   */
  public AttributeSpec<T> withParam(String name, Object value) {
    return AttributeSpecBuilder.create(this)
      .params().set(name, value).done()
      .build();
  }

  /**
   * Returns a new attribute spec with parameters replaced with a new map.
   *
   * @param newParams replacement parameters
   */
  public AttributeSpec<T> replaceParams(Map<String, Object> newParams) {
    return new AttributeSpec<>(getId(), getFormat(), newParams, true);
  }

  /**
   * Returns a new attribute spec with all parameters removed.
   */
  public AttributeSpec<T> noParams() {
    return new AttributeSpec<>(getId(), getFormat());
  }

  /**
   * Verifies and converts {@link AttributeLoader} to another type with a different value format.
   */
  public AttributeLoader<T> cast(AttributeLoader<?> loader) {
    if (loader == null) return null;
    if (!getFormat().getValueClass().isAssignableFrom(loader.getAttributeSpec().getFormat().getValueClass())) {
      throw new ClassCastException("cannot cast " + loader + " to " + this);
    }
    //noinspection unchecked
    return (AttributeLoader<T>) loader;
  }

  public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;

    AttributeSpec that = (AttributeSpec) o;

    if (!myFormat.equals(that.myFormat)) return false;
    if (!myId.equals(that.myId)) return false;
    if (!myParams.equals(that.myParams)) return false;

    return true;
  }

  public int hashCode() {
    int result = myHashCode;
    if (result == 0) {
      result = myId.hashCode();
      result = 31 * result + myFormat.hashCode();
      result = 31 * result + (myParams.hashCode());
      myHashCode = result;
    }
    return result;
  }

  public String toString() {
    StringBuilder builder = new StringBuilder();
    builder.append(myId);
    if (!myParams.isEmpty()) {
      builder.append(':').append(StructureUtil.toJson(myParams));
    }
    builder.append(':').append(myFormat);
    return builder.toString();
  }
}
