package com.almworks.jira.structure.api.permissions;

import com.atlassian.annotations.PublicApi;
import com.google.common.collect.ImmutableSet;
import org.jetbrains.annotations.Nullable;

import java.util.Set;

/**
 * Core app-level permissions checked by Structure.
 * @see com.almworks.jira.structure.api.settings.StructureConfiguration
 */
@PublicApi
public interface CoreAppPermissions {
  /**
   * Permission to use Structure.
   */
  StructureAppPermission USE = new StructureAppPermission("use");

  /**
   * Permission to create new structures.
   *
   * The user must also have the {@link #USE} permission.
   */
  StructureAppPermission CREATE_STRUCTURE = new StructureAppPermission("createStructure");

  /**
   * Permission to configure and control synchronizers.
   *
   * The user must also have the {@link #USE} permission and
   * {@link PermissionLevel#ADMIN} access to the structure.
   */
  StructureAppPermission SYNCHRONIZATION = new StructureAppPermission("synchronization");

  /**
   * Permission to use automation (generators and effectors).
   *
   * The user must also have the {@link #USE} permission and
   * {@link PermissionLevel#AUTOMATE} access to the structure.
   */
  StructureAppPermission AUTOMATION = new StructureAppPermission("automation");

  /**
   * Permission to create and configure generators.
   *
   * The user must also have the {@link #USE} and {@link #AUTOMATION}
   * permissions and {@link PermissionLevel#AUTOMATE} access to the structure.
   */
  StructureAppPermission CONFIGURE_GENERATORS = new StructureAppPermission("configureGenerators");

  /**
   * Permission to create and configure effectors.
   *
   * The user must also have the {@link #USE} and {@link #AUTOMATION}
   * permissions and {@link PermissionLevel#AUTOMATE} access to the structure.
   */
  StructureAppPermission CONFIGURE_EFFECTORS = new StructureAppPermission("configureEffectors");

  /**
   * Permission to execute effectors installed in structures.
   *
   * The user must also have the {@link #USE} permission,
   * the {@link com.atlassian.jira.permission.GlobalPermissionKey#BULK_CHANGE}
   * Jira permission, and {@link PermissionLevel#VIEW} access to the structure.
   */
  StructureAppPermission EXECUTE_EFFECTORS = new StructureAppPermission("executeEffectors");

  /**
   * Permission to execute effectors on query results.
   *
   * The user must also have the {@link #USE}, {@link #AUTOMATION},
   * {@link #CONFIGURE_EFFECTORS}, and {@link #EXECUTE_EFFECTORS} permissions
   * and the {@link com.atlassian.jira.permission.GlobalPermissionKey#BULK_CHANGE}
   * Jira permission.
   */
  StructureAppPermission EXECUTE_EFFECTORS_ON_QUERIES = new StructureAppPermission("executeEffectorsOnQueries");

  Set<StructureAppPermission> ALL = ImmutableSet.copyOf(new StructureAppPermission[] {
    USE, CREATE_STRUCTURE, SYNCHRONIZATION, AUTOMATION, CONFIGURE_GENERATORS,
    CONFIGURE_EFFECTORS, EXECUTE_EFFECTORS, EXECUTE_EFFECTORS_ON_QUERIES
  });

  @Nullable
  static StructureAppPermission byKey(@Nullable String key) {
    return ALL.stream().filter(p -> p.getKey().equals(key)).findAny().orElse(null);
  }
}
