package com.almworks.jira.structure.api.item;

import com.almworks.jira.structure.api.pull.DataVersion;
import com.almworks.jira.structure.api.pull.VersionedDataSource;
import org.jetbrains.annotations.NotNull;

/**
 * <p>{@code ItemTracker} service provides means to track and report changes to items.</p>
 *
 * <h3>Tracking</h3>
 *
 * <p>Item tracking is pull-based. Client can request which items have been updated since the
 * last time it checked, using version information provided with the previous reply from ItemTracker.</p>
 *
 * <p>The response may be an incremental update, which lists specific items that were changed,
 * or it may be a full update, which means that possibly any items could have changed. Full update means
 * that the client code needs to recalculate everything that is based on any items, while incremental update
 * allows the client code to recalculate only the parts that are invalidated by the changes.</p>
 *
 * <p>The version is tracked with two integers - {@code signature} and {@code version}. Version is a
 * number that is incremented for every reported item. Signature defines the "id space" for versions -
 * versions under different signatures are not comparable and require full update. Signature is needed because
 * the lifespan of the client may exceed the lifespan of the ItemTracker (for example, a remote client), and
 * version number is not persisted - so in the next incarnation, the ItemTracker will have versions starting from 1 again.</p>
 *
 * <p>Note that it is not guaranteed that signature stays the same during lifespan of a single instance of ItemTracker,
 * however, it may be so depending on the implementation.</p>
 *
 * <h3>Reporting</h3>
 *
 * <p>To report changes, simply call {@link #recordChanges} method with the changed item IDs.</p>
 *
 * <p>Note that this component is passive. It does not listen to anything nor does it track the changes to issues
 * or other items. In order to be useful, there should be some other components that listen to changes and
 * report them to the ItemTracker.</p>
 *
 * <p>Structure contains an active component that captures most of the events that happen to issues and routes them
 * into the ItemTracker. However, you can report updates to issues or to other items at any time if you have anything
 * calculated for those items (whether attributes or dynamic forest) that needs to be recalculated.</p>
 *
 * <p>When especially large set of changes is reported, ItemTracker may dismiss the detailed changes and force
 * a full update for all clients instead.</p>
 *
 * <h3>Cluster-Wide Service</h3>
 *
 * <p>On JIRA Data Center, this service works across the cluster - changes reported on one node will
 * eventually be seen by the clients pulling updates on all nodes.</p>
 *
 * <p>It is <strong>not</strong> guaranteed that the order of the changed items will be the same on all nodes.</p>
 *
 * <p><strong>NB:</strong> The signatures of ItemTracker instances on different nodes will be different,
 * so the efficiency of using ItemTracker very much depends on the session affinity: if the same client is redirected
 * to another node for updates, it will result in a full update and a change of signature.</p>
 *
 * @see ItemVersionUpdate
 */
public interface ItemTracker extends VersionedDataSource<ItemVersionUpdate> {
  /**
   * <p>Provides an update on the tracked items, based on the previously seen signature and version.</p>
   *
   * <p>The response may be an empty update (no changes), full update (unknown changes - you need to recalculate
   * everything), or incremental update (specific items have changed).</p>
   *
   * <p>To make the first call, use {@code DataVersion.ZERO}. That will result in a full update.</p>
   *
   * @param fromVersion last seen version with signature
   * @return empty, full or incremental update to the current version
   * @see ItemVersionUpdate
   */
  @NotNull
  ItemVersionUpdate getUpdate(@NotNull DataVersion fromVersion);

  /**
   * Convenience method for calling {@link #recordChanges} for a single item.
   *
   * @param id item ID
   */
  void recordChange(ItemIdentity id);

  /**
   * Records changes of the specified items. All clients calling {@link #getUpdate} afterwards will either receive
   * an incremental update with these {@code ids}, or a full update.
   *
   * @param ids changed items IDs
   */
  void recordChanges(Iterable<ItemIdentity> ids);

  /**
   * Resets ItemTracker. All clients calling {@link #getUpdate} afterwards will receive full update.
   */
  void reset();
}
